package ru.yandex.partner.libs.bs.json;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.deser.DeserializationProblemHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BkDataDeserializationProblemHandler extends DeserializationProblemHandler {
    private static final Logger LOG = LoggerFactory.getLogger(BkDataDeserializationProblemHandler.class);
    public static final Set<String> VALID_BOOLEAN_STRINGS =
            Set.of(JsonToken.VALUE_FALSE.asString(), JsonToken.VALUE_TRUE.asString());

    private final Map<List<String>, String> errorsCollector;

    public BkDataDeserializationProblemHandler(Map<List<String>, String> errorsCollector) {
        this.errorsCollector = errorsCollector;
    }

    /**
     * Also tries to fix boolean values
     */
    @Override
    public Object handleWeirdStringValue(DeserializationContext ctxt, Class<?> targetType,
                                         String str, String failureMsg) throws IOException {
        if (targetType != boolean.class && targetType != Boolean.class) {
            return handleParseError(ctxt, failureMsg);
        }

        if (str.isEmpty()) {
            return handleParseError(ctxt, "Empty string can't be parsed as boolean");
        }

        if (Character.isAlphabetic(str.charAt(0))) {
            if (VALID_BOOLEAN_STRINGS.contains(str)) {
                return Boolean.parseBoolean(str);
            } else {
                return handleParseError(ctxt, failureMsg);
            }
        } else {
            try {
                return Integer.parseInt(str) != 0;
            } catch (RuntimeException e) {
                return handleParseError(ctxt, "Invalid boolean string");
            }
        }
    }

    private Object handleParseError(DeserializationContext ctxt, String msg) {
        var jsonStreamContext = ctxt.getParser().getParsingContext();

        String strPath = jsonStreamContext.pathAsPointer().toString()
                .replaceAll("^/+", "")
                .replaceAll("/+$", "");
        List<String> path;
        if (strPath.isEmpty()) {
            path = List.of();
        } else {
            path = List.of(strPath.split("/"));
        }
        errorsCollector.put(path, msg);
        return null;
    }

    @Override
    public boolean handleUnknownProperty(DeserializationContext ctxt, JsonParser p,
                                         JsonDeserializer<?> deserializer, Object beanOrClass,
                                         String propertyName) throws IOException {
        LOG.info(
                "Ignored unknown property in bkdata: {}/{}",
                ctxt.getParser().getParsingContext().pathAsPointer(),
                propertyName
        );
        p.skipChildren();
        return true;
    }
}
