package ru.yandex.partner.libs.bs.json;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.Function;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Класс для парных конвертаций объектов туда-обратно
 */
public interface ReversibleConversion<T> {
    Logger LOGGER = LoggerFactory.getLogger(ReversibleConversion.class);

    T forward(T node);

    T backward(T node);

    /**
     * <p>
     * Прямая и обратная конвертации для кейса, когда имеем json объект с ключами-идентификаторами,
     * а в значении пейлоад, а нам нужен массив (ключ переезжает в элемент массива).
     * </p>
     * <p>Например, было:
     * <pre>
     * "AdTypeSet": {
     *     "media-performance": 1,
     *     "video-motion": 0,
     *     "text": 1,
     *     "media": 1,
     *     "video-performance": 0,
     *     "video": 0
     * }
     * </pre>
     * Стало:
     * <pre>
     * "AdTypeSet":[
     *     {"Value":1,"AdType":"MEDIA_PERFORMANCE"},
     *     {"Value":0,"AdType":"VIDEO_MOTION"},
     *     {"Value":1,"AdType":"TEXT"},
     *     {"Value":1,"AdType":"MEDIA"},
     *     {"Value":0,"AdType":"VIDEO_PERFORMANCE"},
     *     {"Value":0,"AdType":"VIDEO"}
     * ]
     * </pre>
     * Здесь также конвертируются исходные ключи и значения
     * </p>
     *
     * @return исходный объект после мутаций
     */
    static ReversibleConversion<JsonNode> objectToArrayWithId(
            JsonNodeFactory nf,
            String field,
            String idFieldName,
            ReversibleConversion<String> keyToIdValue,
            ReversibleConversion<JsonNode> convertValue
    ) {
        return new ReversibleConversion<>() {
            @Override
            public JsonNode forward(JsonNode node) {
                var objectNode = node.path(field);
                if (objectNode.isMissingNode()) {
                    return node;
                }
                if (!objectNode.isObject()) {
                    return node;
                }
                var resultArrayNode = nf.arrayNode(objectNode.size());
                objectNode.fields().forEachRemaining(idAndPayload -> {
                    var geoObj = (ObjectNode) convertValue.forward(idAndPayload.getValue());
                    geoObj.put(idFieldName, keyToIdValue.forward(idAndPayload.getKey()));
                    resultArrayNode.add(geoObj);
                });
                ((ObjectNode) node).replace(field, resultArrayNode);
                return node;
            }

            @Override
            public JsonNode backward(JsonNode node) {
                var arrayNode = node.path(field);
                if (arrayNode.isMissingNode()) {
                    return node;
                }
                if (!arrayNode.isArray()) {
                    return node;
                }
                var resultObj = nf.objectNode();
                for (JsonNode jsonNode : arrayNode) {
                    var idValue = jsonNode.get(idFieldName);
                    if (jsonNode.isObject()) {
                        ((ObjectNode) jsonNode).remove(idFieldName);
                    }
                    // mobile blocks in bk_log are missing BLockID
                    if (idValue != null) {
                        resultObj.replace(keyToIdValue.backward(idValue.asText()), convertValue.backward(jsonNode));
                    }
                }
                ((ObjectNode) node).replace(field, resultObj);
                return node;
            }
        };
    }

    static ReversibleConversion<JsonNode> objectToArrayWithId(
            JsonNodeFactory nodeFactory,
            String field,
            String idFieldName
    ) {
        return objectToArrayWithId(nodeFactory, field, idFieldName,
                ReversibleConversion.identity(),
                ReversibleConversion.identity()
        );
    }

    static <T> ReversibleConversion<T> identity() {
        return new ReversibleConversion<T>() {
            @Override
            public T forward(T node) {
                return node;
            }

            @Override
            public T backward(T node) {
                return node;
            }
        };
    }

    @SafeVarargs
    static <T> ReversibleConversion<T> combine(ReversibleConversion<T>... conversions) {
        var reversedConversionsList = new ArrayList<>(List.of(conversions));
        Collections.reverse(reversedConversionsList);
        return new ReversibleConversion<>() {
            @Override
            public T forward(T node) {
                for (ReversibleConversion<T> conversion : conversions) {
                    node = conversion.forward(node);
                }
                return node;
            }

            @Override
            public T backward(T node) {
                for (ReversibleConversion<T> conversion : reversedConversionsList) {
                    node = conversion.backward(node);
                }
                return node;
            }
        };
    }

    static <T> ReversibleConversion<T> simply(Function<T, T> forward, Function<T, T> backward) {
        return new ReversibleConversion<T>() {
            @Override
            public T forward(T node) {
                return forward.apply(node);
            }

            @Override
            public T backward(T node) {
                return backward.apply(node);
            }
        };
    }

    static ReversibleConversion<JsonNode> jsonifyKey(ObjectMapper om, String key) {
        return new ReversibleConversion<>() {
            @Override
            public JsonNode forward(JsonNode node) {
                var dataNode = node.path(key);
                if (!dataNode.isMissingNode()) {
                    ((ObjectNode) node).put(key, dataNode.toString());
                }
                return node;
            }

            @Override
            public JsonNode backward(JsonNode node) {
                var dataNode = node.path(key);
                if (!dataNode.isMissingNode() && dataNode.isTextual()) {
                    ((ObjectNode) node).replace(key, parseSneaky(om, dataNode.asText()));
                }
                return node;
            }
        };
    }

    static ReversibleConversion<JsonNode> mapKey(String key, String newKey) {
        return mapKey(key, newKey, false);
    }

    static ReversibleConversion<JsonNode> removeKey(String key) {
        return simply(node -> {
            ((ObjectNode) node).remove(key);
            return node;
        }, Function.identity());
    }

    static ReversibleConversion<JsonNode> mapKey(String key, String newKey, boolean enableBackward) {
        return new ReversibleConversion<>() {
            @Override
            public JsonNode forward(JsonNode node) {
                if (!node.isObject()) {
                    return node;
                }
                return moveKey((ObjectNode) node, key, newKey);
            }

            @NotNull
            private static ObjectNode moveKey(ObjectNode node, String from, String to) {
                var keyValue = node.remove(from);
                if (keyValue != null) {
                    node.replace(to, keyValue);
                }
                return node;
            }

            @Override
            public JsonNode backward(JsonNode node) {
                if (!enableBackward) {
                    return node;
                }
                if (!node.isObject()) {
                    return node;
                }
                return moveKey((ObjectNode) node, newKey, key);
            }
        };
    }

    @SafeVarargs
    static ReversibleConversion<JsonNode> each(ReversibleConversion<JsonNode>... convert) {
        var converter = combine(convert);
        return new ReversibleConversion<>() {
            @Override
            public JsonNode forward(JsonNode node) {
                if (!node.isContainerNode()) {
                    return node;
                }
                for (Object item : node) {
                    converter.forward((JsonNode) item);
                }
                return node;

            }

            @Override
            public JsonNode backward(JsonNode node) {
                if (!node.isContainerNode()) {
                    return node;
                }
                for (Object item : node) {
                    converter.backward((JsonNode) item);
                }
                return node;
            }
        };
    }

    static ReversibleConversion<JsonNode> onPath(
            String key,
            ReversibleConversion<JsonNode> convert
    ) {
        return simply(
                node -> {
                    var subNode = node.path(key);
                    if (subNode.isMissingNode()) {
                        return node;
                    }
                    ((ObjectNode) node).replace(key, convert.forward(subNode));
                    return node;
                },
                node -> {
                    var subNode = node.path(key);
                    if (subNode.isMissingNode()) {
                        return node;
                    }
                    ((ObjectNode) node).replace(key, convert.backward(subNode));
                    return node;
                }
        );
    }

    /**
     * Notice - такой проход позволяет только мутацию узла, но не его замену
     */
    static ReversibleConversion<JsonNode> onPath(
            Function<JsonNode, JsonNode> walker,
            ReversibleConversion<JsonNode> convert
    ) {
        return new ReversibleConversion<>() {
            @Override
            public JsonNode forward(JsonNode node) {
                var subNode = walker.apply(node);
                if (subNode.isMissingNode()) {
                    return node;
                }
                convert.forward(subNode);
                return node;
            }

            @Override
            public JsonNode backward(JsonNode node) {
                var subNode = walker.apply(node);
                if (subNode.isMissingNode()) {
                    return node;
                }
                convert.backward(subNode);
                return node;
            }
        };
    }

    private static JsonNode parseSneaky(ObjectMapper om, String bkDataJson) {
        try {
            return om.readTree(bkDataJson);
        } catch (JsonProcessingException e) {
            throw new RuntimeException("Could not parse BK_DATA", e);
        }
    }
}
