package ru.yandex.partner.libs.annotation;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

/**
 * Аннотация, которой нужно помечать методы, которые должны вызываться в транзакции.
 * <p>
 * ВАЖНО!
 * <p>
 * Spring AOP накладывает ограничения на классы и методы, к которым применяются аспекты.
 * Применительно к данной аннотации, нужно помнить о следующем:
 * <p>
 * 1) Объекты, использующие эту аннотацию, должны внедряться ПО ИНТЕРФЕЙСУ, не по конкретному классу.
 * 2) Транзакционные методы должны быть public.
 * 3) Вызывать эти методы нужно снаружи (из другого объекта), иначе транзакция открыта не будет.
 *
 * Так же нужно помнить, что работают транзакции только если в одной из конфигураций объявлен
 * бин типа PlatformTransactionManager и добавлена аннотация @EnableTransactionManagement
 * (см. MysqlConfiguration)
 */
@Target({ElementType.METHOD, ElementType.TYPE})
@Retention(RetentionPolicy.RUNTIME)
@Transactional(
        // откатываем транзакцию при любом исключении
        rollbackFor = Exception.class,
        // транзакции вложенные, то есть, если одна открыта внутри другой, то можно её откатить, не откатывая первую
        propagation = Propagation.NESTED
)
public @interface PartnerTransactional {
}
