package ru.yandex.partner.libs.common;

import java.io.IOException;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;

import com.sun.jdi.request.InvalidRequestStateException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.web.server.ResponseStatusException;

import static com.google.common.base.Preconditions.checkNotNull;

public class HttpUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(HttpUtils.class);

    private HttpUtils() {
    }

    /**
     * Пытается получить реальный IP с учетом хедера X-Real-IP
     *
     * @param request http запрос
     * @return строка с IP адресом
     */
    public static String getIpAddrFromRequest(HttpServletRequest request) {
        checkNotNull(request);
        String ip = request.getHeader("X-Real-IP");
        if (ip != null && !"".equals(ip.trim()) && !"unknown".equalsIgnoreCase(ip)) {
            return ip;
        }
        return request.getRemoteAddr();
    }

    /**
     * Пытается получить хост по хедеру Host
     *
     * @param request http запрос
     * @return строка с хостом или IP
     * @throws InvalidRequestStateException если нет заголовка Host или он некорректен
     */
    public static String getServerHostFromRequest(HttpServletRequest request) {
        checkNotNull(request);
        String headerName = "Host";
        String ip = request.getHeader(headerName);
        if (ip != null && !"".equals(ip.trim()) && !"unknown".equalsIgnoreCase(ip)) {
            return ip;
        }
        throw new InvalidRequestStateException("Header missing or invalid. HeaderName = " + headerName);
    }

    /**
     * Выполняет Http запрос с повторами при ошибках
     *
     * @param client           - HttpClient
     * @param request          - HttpRequest
     * @param retryCount       - Количество повторов
     * @param msBetweenRetries - время между повторами, мс
     * @return - Optional<HttpResponse<String>>
     */
    public static <T> Optional<HttpResponse<T>> sendRequestWithRetries(
            HttpClient client,
            HttpRequest request,
            HttpResponse.BodyHandler<T> bodyHandler,
            int retryCount,
            int msBetweenRetries
    ) {
        checkNotNull(client);
        checkNotNull(request);

        for (int i = 0; i < retryCount; i++) {
            try {
                HttpResponse<T> response = client.send(request, bodyHandler);
                int statusCode = response.statusCode();
                if (statusCode >= 200 && statusCode < 500) {
                    return Optional.of(response);
                }
                Thread.sleep(msBetweenRetries);
            } catch (IOException e) {
                LOGGER.warn("IOException during HTTP request", e);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException("Interrupted exception during HTTP request", e);
            }
            LOGGER.warn("Retry HTTP request");
        }
        LOGGER.debug("Error receiving response from: {}", request.uri());
        return Optional.empty();
    }

    public static Optional<HttpResponse<String>> sendRequestWithRetries(
            HttpClient client,
            HttpRequest request,
            int retryCount
    ) {
        return sendRequestWithRetries(client, request, HttpResponse.BodyHandlers.ofString(), retryCount, 1000);
    }

    public static Optional<HttpResponse<String>> sendRequestWithRetries(
            HttpClient client,
            HttpRequest request
    ) {
        return sendRequestWithRetries(client, request, HttpResponse.BodyHandlers.ofString());
    }

    public static <T> Optional<HttpResponse<T>> sendRequestWithRetries(
            HttpClient client,
            HttpRequest request,
            HttpResponse.BodyHandler<T> bodyHandler
    ) {
        return sendRequestWithRetries(client, request, bodyHandler, 3, 1000);
    }

    public static Map<String, List<String>> getHeadersMapFromRequest(HttpServletRequest request) {
        checkNotNull(request);
        try {
            return Collections.list(request.getHeaderNames()).stream()
                    .collect(
                            Collectors.toUnmodifiableMap(
                                    k -> k,
                                    k -> Collections.list(request.getHeaders(k))
                            )
                    );
        } catch (RuntimeException e) {
            throw new ResponseStatusException(HttpStatus.BAD_REQUEST, e.getMessage(), e);
        }

    }

    public static Map<String, List<String>> getParametersMapFromRequest(HttpServletRequest request) {
        checkNotNull(request);
        return Collections.list(request.getParameterNames()).stream()
                .collect(
                        Collectors.toUnmodifiableMap(
                                k -> k,
                                k -> Arrays.asList(request.getParameterValues(k))
                        )
                );
    }
}
