package ru.yandex.partner.libs.common;

import java.util.function.UnaryOperator;

import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import reactor.core.publisher.Mono;
import reactor.util.retry.Retry;

import ru.yandex.partner.defaultconfiguration.rpc.RpcConfigProps;
import ru.yandex.partner.libs.exceptions.HttpErrorStatusEnum;
import ru.yandex.partner.libs.exceptions.I18nResponseStatusException;

public class PartnerRequestService {

    private PartnerRequestService() { }

    public static <T> UnaryOperator<Mono<ResponseEntity<T>>> retryWithFixedDelay(RpcConfigProps config) {
        return mono -> mono
                .retryWhen(Retry.fixedDelay(config.getRetries(), config.getDelay())
                        .onRetryExhaustedThrow((spec, sig) -> sig.failure())
                )
                .timeout(config.getTimeout())
                .transform(addErrorMapping());
    }

    public static <T> UnaryOperator<Mono<ResponseEntity<T>>> retryWithBackoff(RpcConfigProps config) {
        return mono -> mono
                .retryWhen(Retry.backoff(config.getRetries(), config.getDelay())
                        .onRetryExhaustedThrow((spec, sig) -> sig.failure())
                )
                .timeout(config.getTimeout())
                .transform(addErrorMapping());
    }

    public static <T> ResponseEntity<T> serviceUnavailable() {
        return ResponseEntity.status(HttpStatus.SERVICE_UNAVAILABLE).build();
    }

    private static <T> UnaryOperator<Mono<ResponseEntity<T>>> addErrorMapping() {
        return mono -> mono
//              для произвольной ошибки обобщенная 5xx
                .onErrorMap(e -> new I18nResponseStatusException(HttpErrorStatusEnum.ERROR__INTERNAL, e));
    }
}

