package ru.yandex.partner.libs.parser;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.mariuszgromada.math.mxparser.Expression;

import ru.yandex.partner.libs.parser.exceptions.ParserException;
import ru.yandex.partner.libs.parser.models.MultiStateInterface;

public class Parser {
    private final MultiStateInterface multiStateInterface;
    private final byte msLengths;
    private final byte[] bitsArr;
    private final boolean[] bitsUsedInExpression;

    private int realNumber;
    private boolean isUsedBitModified;

    public Parser(MultiStateInterface multiStateInterface) {
        this.multiStateInterface = multiStateInterface;
        msLengths = multiStateInterface.length();
        bitsArr = new byte[msLengths];
        bitsUsedInExpression = new boolean[msLengths];
    }

    public List<Integer> getAllNumbersForExpression(String expString) {
        expString = expString.toUpperCase();
        final Expression expression = prepareExpression(expString);
        final ArrayList<Integer> resultList = new ArrayList<>();
        boolean lastResult = false;
        int possibleCombinations = (int) Math.pow(2, msLengths);

        isUsedBitModified = true;
        for (int i = 0; i < possibleCombinations; i++) {
            if (isUsedBitModified) {
                resetArguments(expression);
                double calculationResult = expression.calculate();
                if (Double.isNaN(calculationResult)) {
                    throw new ParserException("Formula parsing failed " + expString);
                } else if (calculationResult == 1.0) {
                    lastResult = true;
                    resultList.add(realNumber);
                } else {
                    lastResult = false;
                }
            } else if (lastResult) {
                resultList.add(realNumber);
            }
            incrementMask();
        }
        return resultList;
    }

    public String getBitsExpression(String expString) {
        expString = expString.toUpperCase();
        Expression expression = prepareExpression(expString);
        double calculate = expression.calculate();
        if (Double.isNaN(calculate)) {
            throw new ParserException("Formula parsing failed " + expString);
        }
        for (String expressionArg : getAllArgs(expression)) {
            expString = expString.replace(expressionArg,
                    "(multistate & " + multiStateInterface.getByteMaskByName(expressionArg) + ")");
        }
        return expString;
    }

    private Expression prepareExpression(String expString) {
        expString = expString
                .replace(" AND ", " & ")
                .replace(" OR ", " | ")
                .replace(" NOT ", " ~ ")
                .replace("(NOT ", "(~")
                .replace("NOT(", "~(")
                .replace("NOT ", "~");

        Expression expression = new Expression(expString);
        final String[] allArgs = getAllArgs(expression);
        expression.defineArguments(allArgs);
        resetArguments(expression);

        if (!expression.checkSyntax()) {
            throw new ParserException("Formula syntax exception " + expString);
        }
        if (expression.getMissingUserDefinedArguments() != null && expression.getMissingUserDefinedArguments().length > 0) {
            throw new ParserException("Not all arguments defined "
                    + Arrays.toString(expression.getMissingUserDefinedArguments()));
        }
        return expression;
    }

    private String[] getAllArgs(Expression expression) {
        final String[] tokens = expression.getCopyOfInitialTokens()
                .stream()
                .map(token -> token.tokenStr)
                .filter(s -> !s.equals("&")
                        && !s.equals("|")
                        && !s.equals("~")
                        && !s.equals(")")
                        && !s.equals("("))
                .toArray(String[]::new);
        for (String item : tokens) {
            try {
                bitsUsedInExpression[multiStateInterface.getBytePositionByName(item)] = true;
            } catch (IllegalArgumentException e) {
                throw new ParserException("Cant find argument " + item + "\n" + expression.getExpressionString());
            }
        }
        return tokens;
    }

    private void resetArguments(Expression expression) {
        for (int i = 0; i < expression.getArgumentsNumber(); i++) {
            final String argumentName = expression.getArgument(i).getArgumentName();
            expression.setArgumentValue(argumentName, bitsArr[multiStateInterface.getBytePositionByName(argumentName)]);
        }
    }

    private void incrementMask() {
        realNumber++;
        isUsedBitModified = false;
        int index = bitsArr.length - 1;
        while (true) {
            if (index == -1) {
                Arrays.fill(bitsArr, (byte) 0);
                return;
            }
            isUsedBitModified = isUsedBitModified || bitsUsedInExpression[index];
            if (bitsArr[index] == 0) {
                bitsArr[index] = 1;
                return;
            } else {
                bitsArr[index] = 0;
                index--;
            }
        }
    }
}
