package ru.yandex.partner.core.block;

import java.util.Map;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.jetbrains.annotations.NotNull;

import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.ContentBlock;
import ru.yandex.partner.core.entity.block.model.InternalMobileRtbBlock;
import ru.yandex.partner.core.entity.block.model.InternalRtbBlock;
import ru.yandex.partner.core.entity.block.model.MobileRtbBlock;
import ru.yandex.partner.core.entity.block.model.RtbBlock;

public class BlockUniqueIdConverter {
    public enum Prefixes {
//https://wiki.yandex-team.ru/partner/w/dev/infrastructure/systems/applications/partner/java/blockuniqueid/prefix-table/
        CONTEXT_ON_SITE_RTB_PREFIX(0, "R-A-", RtbBlock.class),
        INTERNAL_CONTEXT_ON_SITE_RTB_PREFIX(1, "R-I-", InternalRtbBlock.class),
        MOBILE_RTB_PREFIX(2, "R-M-", MobileRtbBlock.class),
        CONTEXT_ON_SITE_CONTENT_PREFIX(3, "C-A-", ContentBlock.class),
        INTERNAL_MOBILE_RTB_PREFIX(4, "R-IM-", InternalMobileRtbBlock.class);

        private static final Map<Long, Prefixes> PREFIXES_BY_NUMBER = Stream.of(values())
                .collect(Collectors.toMap(Prefixes::getPrefixNumber, Function.identity()));

        private static final Map<String, Long> NUMBER_BY_STRING = Stream.of(values())
                .collect(Collectors.toMap(Prefixes::getPrefixString, Prefixes::getPrefixNumber));

        private static final Map<Class<?>, Prefixes> PREFIXES_BY_CLASS = Stream.of(values())
                .collect(Collectors.toMap(Prefixes::getModelClass, Function.identity()));

        private final long prefixNumber;
        private final String prefixString;
        private final Class<?> modelClass;

        Prefixes(long prefixNumber, String prefixString, Class<?> modelClass) {
            this.prefixNumber = prefixNumber;
            this.prefixString = prefixString;
            this.modelClass = modelClass;
        }

        public long getPrefixNumber() {
            return prefixNumber;
        }

        public String getPrefixString() {
            return prefixString;
        }

        public Class<?> getModelClass() {
            return modelClass;
        }

        /**
         * @return number if prefix exists else null
         */
        public static Long numberByString(String prefixString) {
            return NUMBER_BY_STRING.get(prefixString);
        }

        /**
         * @return number if prefix exists else null
         */
        public static Long numberByClass(Class<?> modelClass) {
            Prefixes prefixes = PREFIXES_BY_CLASS.get(modelClass);

            return prefixes != null
                    ? prefixes.getPrefixNumber()
                    : null;
        }

        public static Prefixes byClass(@NotNull Class<? extends BaseBlock> modelClass) {
            return PREFIXES_BY_CLASS.get(modelClass);
        }

        public static Prefixes byUniqueId(long uniqueId) {
            return PREFIXES_BY_NUMBER.get(prefixFromUniqueId(uniqueId));
        }

        /**
         * @return string if prefix exists else null
         */
        public static String stringByNumber(long prefixNumber) {
            Prefixes prefixes = PREFIXES_BY_NUMBER.get(prefixNumber);
            return prefixes == null ? null : prefixes.getPrefixString();
        }
    }

    private static final long PREFIX_BITMASK = 255; // 8 bits
    private static final long PREFIX_SHIFT = 55;
    private static final long PAGE_ID_BITMASK = 4_294_967_295L; // 32 bits
    private static final long PAGE_ID_SHIFT = 23;
    private static final long BLOCK_ID_BITMASK = 8_388_607L; // 23 bits
    private static final long BLOCK_ID_SHIFT = 0;

    private static final Pattern PATTERN = Pattern.compile("([^\\d]+)(\\d+)-(\\d+)");

    /**
     * @param uniqueId uniqueId
     * @return example: R-A-400123-3
     */
    public static String convertToPublicId(long uniqueId) {
        return convertToPublicId(
                prefixFromUniqueId(uniqueId),
                pageIdFromUniqueId(uniqueId),
                blockIdFromUniqueId(uniqueId)
        );
    }


    public static String convertToPublicId(Prefixes prefix, long pageId, long blockId) {
        return convertToPublicId(prefix.getPrefixNumber(), pageId, blockId);
    }

    public static String convertToPublicId(long prefix, long pageId, long blockId) {
        String prefixString = Prefixes.stringByNumber(prefix);

        if (prefixString == null) {
            throw new IllegalArgumentException("Prefix string not found for prefix = " + prefix +
                    ". See " + Prefixes.class);
        }

        return prefixString + pageId + "-" + blockId;
    }

    public static String publicId(BaseBlock block)  {
        return BlockUniqueIdConverter.convertToPublicId(
                BlockUniqueIdConverter.Prefixes.byClass(block.getClass()),
                block.getPageId(),
                block.getBlockId()
        );
    }

    /**
     * @param publicId example: R-A-400123-3
     * @return uniqueId
     */
    public static long convertToUniqueId(String publicId) {
        Matcher matcher = PATTERN.matcher(publicId);


        if (matcher.matches() && matcher.groupCount() == 3) {
            // matcher.group(0) is FullMatch
            String prefixString = matcher.group(1);
            long pageId = Long.parseLong(matcher.group(2));
            long blockId = Long.parseLong(matcher.group(3));

            Long prefixNumber = Prefixes.numberByString(prefixString);

            if (prefixNumber == null) {
                throw new IllegalArgumentException("Prefix not found for public_id = " + publicId +
                        ". See " + Prefixes.class);
            }

            return convertToUniqueId(prefixNumber, pageId, blockId);
        } else {
            throw new IllegalArgumentException("Public_id does not match with '" + PATTERN.pattern()
                    + "'. Public_id = " + publicId);
        }

    }

    public static long convertToUniqueId(Prefixes prefix, long pageId, long blockId) {
        return convertToUniqueId(prefix.getPrefixNumber(), pageId, blockId);
    }
    public static long convertToUniqueId(long prefix, long pageId, long blockId) {
        if (prefix > PREFIX_BITMASK || prefix < 0L
                || pageId > PAGE_ID_BITMASK || pageId < 0L
                || blockId > BLOCK_ID_BITMASK || blockId < 0L
        ) {
            throw new IllegalArgumentException(
                    String.format("Must be: prefix: [0, %d], pageId: [0, %d], blockId: [0, %d]. " +
                                    "Current values: prefix: %d, pageId: %d, blockId: %d",
                            PREFIX_BITMASK, PAGE_ID_BITMASK, BLOCK_ID_BITMASK, prefix, pageId, blockId));
        }

        long modifiedPrefix = (prefix & PREFIX_BITMASK) << PREFIX_SHIFT;
        long modifiedPageId = (pageId & PAGE_ID_BITMASK) << PAGE_ID_SHIFT;
        long modifiedBlockId = (blockId & BLOCK_ID_BITMASK) << BLOCK_ID_SHIFT;
        return modifiedPrefix | modifiedPageId | modifiedBlockId;
    }

    public static long prefixFromUniqueId(long uniqueId) {
        throwIfInvalidUniqueId(uniqueId);
        return (uniqueId >> PREFIX_SHIFT) & PREFIX_BITMASK;
    }

    public static long pageIdFromUniqueId(long uniqueId) {
        throwIfInvalidUniqueId(uniqueId);
        return (uniqueId >> PAGE_ID_SHIFT) & PAGE_ID_BITMASK;
    }

    public static long blockIdFromUniqueId(long uniqueId) {
        throwIfInvalidUniqueId(uniqueId);
        return (uniqueId >> BLOCK_ID_SHIFT) & BLOCK_ID_BITMASK;
    }

    public static String pageIdFromPublicId(String publicId){
        Matcher matcher = PATTERN.matcher(publicId);
        if (matcher.matches() && matcher.groupCount() == 3) {
            return matcher.group(2);
        } else {
            throw new IllegalArgumentException("Public_id does not match with '" + PATTERN.pattern()
                    + "'. Public_id = " + publicId);
        }
    }

    private static void throwIfInvalidUniqueId(long uniqueId) {
        if (uniqueId < 0L) {
            throw new IllegalArgumentException("UniqueId must be positive.  uniqueId = " + uniqueId);
        }
    }

    private BlockUniqueIdConverter() {
        // Utility class
    }
}
