package ru.yandex.partner.core.block;

import java.util.Arrays;
import java.util.EnumSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.partner.core.block.messages.BlockTypeMsg;
import ru.yandex.partner.core.entity.block.model.BlockWithBlockType;
import ru.yandex.partner.libs.i18n.GettextMsg;

public enum MobileBlockType {

    ADAPTIVE_BANNER("adaptive_banner", BlockTypeMsg.ADAPTIVE_BANNER),
    BANNER("banner", BlockTypeMsg.BANNER),
    INTERSTITIAL("interstitial", BlockTypeMsg.INTERSTITIAL),
    NATIVE("native", BlockTypeMsg.NATIVE),
    REWARDED("rewarded", BlockTypeMsg.REWARDED);

    public static final EnumSet<MobileBlockType> SIMPLE_INAPP_BLOCK_TYPES =
            EnumSet.complementOf(EnumSet.of(ADAPTIVE_BANNER));

    public static final EnumSet<MobileBlockType> INTERNAL_SIMPLE_INAPP_BLOCK_TYPES =
            EnumSet.complementOf(EnumSet.of(ADAPTIVE_BANNER, REWARDED));

    private static final Map<String, MobileBlockType> MAP = Arrays.stream(MobileBlockType.values())
            .collect(Collectors.toMap(MobileBlockType::getLiteral, t -> t));

    private final String literal;
    private final GettextMsg msg;

    MobileBlockType(String literal, GettextMsg msg) {
        this.literal = literal;
        this.msg = msg;
    }

    public static Set<String> literals(MobileBlockType... types) {
        return Arrays.stream(types)
                .map(MobileBlockType::getLiteral)
                .collect(Collectors.toSet());
    }

    public static Set<MobileBlockType> forSimpleInapp() {
        return SIMPLE_INAPP_BLOCK_TYPES;
    }

    public static Set<MobileBlockType> forInternalSimpleInapp() {
        return INTERNAL_SIMPLE_INAPP_BLOCK_TYPES;
    }

    public String getLiteral() {
        return literal;
    }

    public GettextMsg getMsg() {
        return msg;
    }

    public boolean hasType(BlockWithBlockType block) {
        return this.getLiteral().equals(block.getBlockType());
    }

    public static MobileBlockType fromLiteral(String blockType) {
        return Optional.ofNullable(blockType)
                .map(MAP::get)
                .orElseThrow(() ->
                        new IllegalArgumentException("There is no mobile block type: %s".formatted(blockType))
                );
    }

    public static MobileBlockType extract(BlockWithBlockType block) {
        return fromLiteral(block.getBlockType());
    }

}
