package ru.yandex.partner.core.block.direct;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import ru.yandex.partner.core.block.MobileBlockType;
import ru.yandex.partner.core.entity.block.model.BlockWithDirectBlock;

public class DirectBlocks {

    private static final List<String> SKIP_FIELDS = List.of(
            "font_size",
            "title_font_size",
            "font_family",
            "callouts",
            "url_background_color"
    );
    private static final List<String> SKIP_FIELDS_WL = Stream.of(SKIP_FIELDS, List.of("limit"))
            .flatMap(Collection::stream).collect(Collectors.toList());

    private static final List<DirectBlock> BLOCKS = List.of(
            new DirectBlock(
                    DirectBlockId._240x400,
                    DirectBlockShortCaptionMsg._240x400,
                    DirectBlockCaptionMsg._240x400,
                    Banners.upTo(2),
                    SKIP_FIELDS
            ),
            new DirectBlock(
                    DirectBlockId._320x100,
                    DirectBlockShortCaptionMsg._320x100,
                    DirectBlockCaptionMsg._320x100,
                    Banners.of(1),
                    SKIP_FIELDS_WL
            ),
            new DirectBlock(
                    DirectBlockId._320x50,
                    DirectBlockShortCaptionMsg._320x50,
                    DirectBlockCaptionMsg._320x50,
                    Banners.of(1),
                    SKIP_FIELDS_WL
            ),
            new DirectBlock(
                    DirectBlockId._300x300,
                    DirectBlockShortCaptionMsg._300x300,
                    DirectBlockCaptionMsg._300x300,
                    Banners.upTo(2),
                    SKIP_FIELDS
            ),
            new DirectBlock(
                    DirectBlockId._300x250,
                    DirectBlockShortCaptionMsg._300x250,
                    DirectBlockCaptionMsg._300x250,
                    Banners.of(1),
                    SKIP_FIELDS_WL
            ),
            new DirectBlock(
                    DirectBlockId._728x90,
                    DirectBlockShortCaptionMsg._728x90,
                    DirectBlockCaptionMsg._728x90,
                    Banners.of(1),
                    SKIP_FIELDS_WL
            ),
            new DirectBlock(
                    DirectBlockId._400x240,
                    DirectBlockShortCaptionMsg._400x240,
                    DirectBlockCaptionMsg._400x240,
                    Banners.upTo(2),
                    SKIP_FIELDS
            ),
            new DirectBlock(
                    DirectBlockId._320x480,
                    DirectBlockShortCaptionMsg._320x480,
                    DirectBlockCaptionMsg._320x480,
                    Banners.of(1),
                    SKIP_FIELDS_WL
            ),
            new DirectBlock(
                    DirectBlockId._480x320,
                    DirectBlockShortCaptionMsg._480x320,
                    DirectBlockCaptionMsg._480x320,
                    Banners.upTo(2),
                    SKIP_FIELDS
            ),
            new DirectBlock(
                    DirectBlockId.HORIZONTAL,
                    DirectBlockShortCaptionMsg.HORIZONTAL,
                    DirectBlockCaptionMsg.HORIZONTAL,
                    Banners.upTo(4),
                    List.of(
                            "font_family",
                            "hover_color",
                            "sitelinks_color",
                            "title_color",
                            "title_font_size",
                            "url_background_color",
                            "url_color"
                    )
            ),
            new DirectBlock(
                    DirectBlockId.VERTICAL,
                    DirectBlockShortCaptionMsg.VERTICAL,
                    DirectBlockCaptionMsg.VERTICAL,
                    Banners.upTo(9)
            ),
            new DirectBlock(
                    DirectBlockId.ADAPTIVE,
                    DirectBlockShortCaptionMsg.ADAPTIVE,
                    DirectBlockCaptionMsg.ADAPTIVE,
                    Banners.upTo(9),
                    SKIP_FIELDS
            ),
            new DirectBlock(
                    DirectBlockId.NATIVE,
                    DirectBlockShortCaptionMsg.NATIVE,
                    DirectBlockCaptionMsg.NATIVE,
                    new Banners(1, 9, 1),
                    List.of(
                            "border_type",
                            "font_family",
                            "font_size",
                            "title_font_size",
                            "site_bg_color",
                            "bg_color",
                            "border_color",
                            "header_bg_color",
                            "title_color",
                            "text_color",
                            "url_color",
                            "url_background_color",
                            "hover_color",
                            "sitelinks_color",
                            "border_radius",
                            "links_underline",
                            "horizontal_align",
                            "callouts"
                    )
            ),
            new DirectBlock(
                    DirectBlockId.ADAPTIVE0418,
                    DirectBlockShortCaptionMsg.ADAPTIVE0418,
                    DirectBlockCaptionMsg.ADAPTIVE0418,
                    Banners.of(1),
                    SKIP_FIELDS_WL
            )
    );

    private static final Map<String, DirectBlock> MAP = BLOCKS.stream()
            .collect(Collectors.toMap(DirectBlock::getName, d -> d));

    private DirectBlocks() { }

    public static List<DirectBlock> all() {
        return BLOCKS;
    }

    public static <M extends BlockWithDirectBlock> List<DirectBlock> getAllowed(M block) {
        return BLOCKS.stream()
                .filter(d -> d.isAllowedBlockType(block))
                .toList();
    }

    public static DirectBlock byName(String name) {
        return MAP.get(name);
    }

    public static String getDefault(String blockType) {
        MobileBlockType type = MobileBlockType.fromLiteral(blockType);
        return switch (type) {
            case NATIVE -> DirectBlockId.NATIVE.getName();
            case INTERSTITIAL, REWARDED -> DirectBlockId._320x480.getName();
            case BANNER -> DirectBlockId._300x250.getName();
            case ADAPTIVE_BANNER -> DirectBlockId.ADAPTIVE0418.getName();
            default -> throw new IllegalArgumentException("Unknown block type: %s".formatted(blockType));
        };
    }
}
