package ru.yandex.partner.core.holder;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.direct.model.ModelProperty;

import static java.util.Objects.requireNonNull;

public class ModelPropertiesHolder {
    // Отображение путей ModelProperty на флаг признака
    // В качестве флага может быть, например, доступность для чтения/редактирования
    private final Map<List<ModelProperty<?, ?>>, Boolean> propertiesMap;

    public ModelPropertiesHolder() {
        this(new HashMap<>());
    }

    public ModelPropertiesHolder(Map<List<ModelProperty<?, ?>>, Boolean> propertiesMap) {
        this.propertiesMap = new HashMap<>(requireNonNull(propertiesMap));
    }

    /**
     * Преобразует Set ModelProperty в ModelPropertyHolder, по умолчанию устанавливает флаг в true
     *
     * @param modelProperties - Set ModelProperty
     * @return - ModelPropertyHolder
     */
    public static ModelPropertiesHolder fromModelProperties(Set<ModelProperty<?, ?>> modelProperties) {
        return fromModelProperties(modelProperties, true);
    }

    public static ModelPropertiesHolder fromModelPropertiesDisabled(Set<ModelProperty<?, ?>> modelProperties) {
        return fromModelProperties(modelProperties, false);
    }

    public static ModelPropertiesHolder fromModelProperties(Set<ModelProperty<?, ?>> modelProperties, boolean flag) {
        ModelPropertiesHolder holder = new ModelPropertiesHolder();
        holder.addPaths(modelProperties, flag);
        return holder;
    }

    /**
     * Добавляет один путь
     * флаг признака не устанавливается
     *
     * @param modelProperties - Один ModelProperty для простого поля
     *                        или список ModelProperty для пути сложного (STRATEGY, MINCPM)
     */
    public Boolean addPath(ModelProperty<?, ?>... modelProperties) {
        return addPath(List.of(modelProperties), true);
    }

    public Boolean addPath(ModelProperty<?, ?> modelProperty, boolean flag) {
        return addPath(List.of(modelProperty), flag);
    }

    public Boolean addPath(List<ModelProperty<?, ?>> modelPropertiesPath, boolean flag) {
        return propertiesMap.put(modelPropertiesPath, flag);
    }

    /**
     * Добавляет пути для набора простых полей
     *
     * @param modelPropertySet - Set ModelProperty для простых полей
     */
    public void addPaths(Set<ModelProperty<?, ?>> modelPropertySet) {
        addPaths(modelPropertySet, true);
    }

    public void addCompositePaths(Set<List<ModelProperty<?, ?>>> modelPropertyPathSet) {
        modelPropertyPathSet.forEach(l -> this.addPath(l, true));
    }

    public void addPaths(Set<ModelProperty<?, ?>> modelPropertySet, boolean flag) {
        requireNonNull(modelPropertySet);
        modelPropertySet.forEach(mp -> this.addPath(mp, flag));
    }

    /**
     * Добавление путей по рассчитанному холдеру вложенной модели
     */
    public void addPaths(ModelProperty<?, ?> prefix, ModelPropertiesHolder subPaths) {
        subPaths.propertiesMap.forEach((path, value) -> {
            var newPath = new ArrayList<ModelProperty<?, ?>>(path.size() + 1);
            newPath.add(prefix);
            newPath.addAll(path);
            propertiesMap.put(
                    newPath,
                    value
            );
        });
    }

    /**
     * Устанавливает пути для набора вложенных ModelProperty для одного outerModelProperty
     *
     * @param outerModelProperty    - внешнее сложное ModelProperty (STRATEGY)
     * @param innerModelPropertySet - вложенные ModelProperty (MINCPM, MEDIA_ACTIVE, ...)
     */
    public void addPathsForCompositeProperty(
            ModelProperty<?, ?> outerModelProperty, Set<ModelProperty<?, ?>> innerModelPropertySet) {
        addPathsForCompositeProperty(outerModelProperty, innerModelPropertySet, true);
    }

    public void addPathsForCompositeProperty(
            ModelProperty<?, ?> outerModelProperty, Set<ModelProperty<?, ?>> innerModelPropertySet, boolean flag) {
        requireNonNull(outerModelProperty);
        requireNonNull(innerModelPropertySet);
        innerModelPropertySet.forEach(mp -> addPath(List.of(outerModelProperty, mp), flag));
    }

    /**
     * Удаляет путь
     *
     * @param modelProperties - один ModelProperty для простого поля
     *                        или список ModelProperty для пути сложного (STRATEGY, MINCPM)
     */
    public Boolean removePath(ModelProperty<?, ?>... modelProperties) {
        return propertiesMap.remove(List.of(modelProperties));
    }

    public void enablePath(ModelProperty<?, ?> modelProperty) {
        this.enablePath(List.of(modelProperty));
    }

    public void enablePath(ModelProperty<?, ?>... modelProperties) {
        this.enablePath(List.of(modelProperties));
    }

    public void enablePath(List<ModelProperty<?, ?>> modelPropertiesPath) {
        propertiesMap.put(modelPropertiesPath, true);
    }

    public void enablePaths(Set<ModelProperty<?, ?>> modelProperties) {
        modelProperties.forEach(this::enablePath);
    }

    public void enablePathsForCompositeProperty(ModelProperty<?, ?> outerModelProperty,
                                                Set<ModelProperty<?, ?>> innerModelPropertySet) {
        innerModelPropertySet.forEach(mp -> enablePath(List.of(outerModelProperty, mp)));
    }

    public void disablePath(ModelProperty<?, ?> modelProperty) {
        this.disablePath(List.of(modelProperty));
    }

    public void disablePath(List<ModelProperty<?, ?>> modelPropertiesPath) {
        propertiesMap.put(modelPropertiesPath, false);
    }

    public void disablePaths(Set<ModelProperty<?, ?>> modelProperties) {
        modelProperties.forEach(this::disablePath);
    }

    public Boolean getFlag(ModelProperty<?, ?> modelProperty) {
        return this.getFlag(List.of(modelProperty));
    }

    public Boolean getFlag(List<ModelProperty<?, ?>> modelProperties) {
        return propertiesMap.get(modelProperties);
    }

    public boolean isEnabled(ModelProperty<?, ?> modelProperty) {
        return isEnabled(List.of(modelProperty));
    }

    public boolean isEnabled(List<ModelProperty<?, ?>> modelProperties) {
        return Boolean.TRUE.equals(propertiesMap.get(modelProperties));
    }

    /**
     * Проверяет наличие конкретного пути
     *
     * @param modelProperties - один ModelProperty для простого поля
     *                        или список ModelProperty для пути сложного (STRATEGY, MINCPM)
     * @return - результат
     */
    public boolean containsPath(ModelProperty<?, ?>... modelProperties) {
        return propertiesMap.containsKey(List.of(modelProperties));
    }

    /**
     * Проверяет наличие хотя бы одного пути из набора простых ModelProperty
     *
     * @param modelProperties - Set ModelProperty для простых полей
     * @return - результат
     */
    public boolean containsAnyPath(Set<ModelProperty<?, ?>> modelProperties) {
        requireNonNull(modelProperties);
        return modelProperties.stream()
                .anyMatch(this::containsPath);
    }

    /**
     * Проверяем наличие сложного ModelProperty
     *
     * @param outerModelProperty - внешнее сложное ModelProperty
     * @return
     */
    public boolean containsCompositePath(ModelProperty<?, ?> outerModelProperty) {
        return propertiesMap.keySet().stream()
                .anyMatch(l -> l.size() > 1 && l.get(0).equals(outerModelProperty));
    }

    /**
     * Получаем все пути сложных вложенных ModelProperty их внешнему ModelProperty
     *
     * @param outerModelProperty - внешнее сложное ModelProperty
     * @return
     */
    public Set<List<ModelProperty<?, ?>>> getNestedPaths(ModelProperty<?, ?> outerModelProperty) {
        return propertiesMap.keySet().stream()
                .filter(l -> l.size() > 1 && l.get(0).equals(outerModelProperty))
                .collect(Collectors.toSet());
    }

    public Set<List<ModelProperty<?, ?>>> getProperties() {
        return new HashSet<>(propertiesMap.keySet());
    }

    public Set<List<ModelProperty<?, ?>>> getEnableProperties() {
        return propertiesMap.entrySet().stream()
                .filter(e -> Boolean.TRUE.equals(e.getValue()))
                .map(Map.Entry::getKey).collect(Collectors.toCollection(HashSet::new));
    }

    protected Map<List<ModelProperty<?, ?>>, Boolean> getPropertiesMap() {
        return propertiesMap;
    }

    public ModelPropertiesHolder merge(ModelPropertiesHolder holder) {
        requireNonNull(holder);
        this.propertiesMap.putAll(holder.getPropertiesMap());
        return this;
    }

    public boolean isEmpty() {
        return propertiesMap.isEmpty();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof ModelPropertiesHolder)) {
            return false;
        }
        ModelPropertiesHolder that = (ModelPropertiesHolder) o;
        return Objects.equals(propertiesMap, that.propertiesMap);
    }

    @Override
    public int hashCode() {
        return Objects.hash(propertiesMap);
    }

    @Override
    public String toString() {
        return "ModelPropertiesHolder{" +
                "propertiesMap=" + propertiesMap +
                '}';
    }
}
