package ru.yandex.partner.core.multistate;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import one.util.streamex.StreamEx;

import ru.yandex.partner.libs.i18n.GettextMsg;

public abstract class AbstractMultistate<T extends StateFlag> implements Multistate<T> {

    private final Map<T, Boolean> flagValues;

    protected AbstractMultistate(long multistateValue) {
        this.flagValues = IntStream.range(0, getStateFlags().size()).boxed()
                .filter(i -> getStateFlags().get(i).isPresent())
                .collect(Collectors.toMap(
                        i -> getStateFlags().get(i).get(), i -> (multistateValue & (1L << i)) != 0L
                ));
    }

    public AbstractMultistate() {
        this(0);
    }

    public AbstractMultistate(Collection<T> enabledFlags) {
        this.flagValues = getStateFlags().stream().filter(Optional::isPresent).map(Optional::get)
                .collect(Collectors.toMap(Function.identity(), enabledFlags::contains));
    }

    @Override
    public boolean hasFlag(T flag) {
        return flag != null && flagValues.getOrDefault(flag, false);
    }

    @Override
    public void setFlags(Map<T, Boolean> values) {
        flagValues.putAll(values);
    }

    @Override
    public List<T> getEnabledFlags() {
        return getStateFlags().stream()
                .filter(Optional::isPresent)
                .map(Optional::get)
                .filter(this::hasFlag)
                .collect(Collectors.toList());
    }

    @Override
    public long toMultistateValue() {
        return IntStream.range(0, getStateFlags().size()).boxed()
                .map(i -> (hasFlag(getStateFlags().get(i).orElse(null)) ? 1L : 0L) << i)
                .reduce(0L, Long::sum);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof AbstractMultistate) || this.getClass() != o.getClass()) {
            return false;
        }
        AbstractMultistate<?> that = (AbstractMultistate<?>) o;
        return this.toMultistateValue() == that.toMultistateValue();
    }

    @Override
    public int hashCode() {
        return Objects.hash(toMultistateValue());
    }

    @Override
    public List<GettextMsg> getNameMessages() {
        List<GettextMsg> messages = getEnabledFlags().stream()
                .filter(flag -> !flag.isPrivate())
                .map(StateFlag::getMsg)
                .collect(Collectors.toList());
        return messages.isEmpty() ? List.of(getEmptyMessage()) : messages;
    }

    @Override
    public String toString() {
        return this.getClass() + ":" +
                "value=" + toMultistateValue() +
                ",name=" + getNameMessages().stream()
                .map(gettextMsg -> gettextMsg.getPayload().getMsg())
                .collect(Collectors.joining(".", "", "."));
    }

    abstract protected List<Optional<T>> getStateFlags();

    abstract protected GettextMsg getEmptyMessage();

    public boolean test(T flag) {
        return hasFlag(flag);
    }

    public boolean test(Predicate<Multistate<T>> expression) {
        return expression.test(this);
    }
}
