package ru.yandex.partner.core.props;

import java.util.Set;

import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;

/**
 * Интерфейс-адаптер для коллекторов изменений моделей
 */
public interface ChangesRegistry<M extends ModelWithId> {
    <V> void process(ModelProperty<? super M, V> property, V value);

    boolean isPropChanged(ModelProperty<? super M, ?> property);

    <V> V getChangedProp(ModelProperty<? super M, V> property);

    static <T extends ModelWithId> ChangesRegistry<T> of(ModelChanges<T> mc) {
        return new ChangesRegistry<>() {
            @Override
            public <V> void process(ModelProperty<? super T, V> property, V value) {
                mc.process(value, property);
            }

            @Override
            public boolean isPropChanged(ModelProperty<? super T, ?> property) {
                return mc.isPropChanged(property);
            }

            @Override
            public <V> V getChangedProp(ModelProperty<? super T, V> property) {
                return mc.getChangedProp(property);
            }
        };
    }

    static <T extends ModelWithId> ChangesRegistry<T> of(AppliedChanges<T> ac) {
        return new ChangesRegistry<>() {
            @Override
            public <V> void process(ModelProperty<? super T, V> property, V value) {
                ac.modify(property, value);
            }

            @Override
            public boolean isPropChanged(ModelProperty<? super T, ?> property) {
                return ac.changed(property);
            }

            @Override
            public <V> V getChangedProp(ModelProperty<? super T, V> property) {
                return ac.getNewValue(property);
            }
        };
    }

    static <T extends ModelWithId> ChangesRegistry<T> of(T model, Set<ModelProperty<?, ?>> modelProperties) {
        return new ChangesRegistry<>() {
            @Override
            public <V> void process(ModelProperty<? super T, V> property, V value) {
                property.set(model, value);
            }

            @Override
            public boolean isPropChanged(ModelProperty<? super T, ?> property) {
                return property.get(model) != null || modelProperties.contains(property);
            }

            @Override
            public <V> V getChangedProp(ModelProperty<? super T, V> property) {
                return (V) property.get(model);
            }
        };
    }

}
