package ru.yandex.partner.core.props;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import one.util.streamex.StreamEx;

import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.partner.core.holder.ModelPropertiesHolder;

import static ru.yandex.partner.core.props.ModelPropertyDefault.forProperty;

/**
 * <p>
 * Схема/метаинформация, описывающая поля модели, дефолтные значениях для них,
 * критерии их появления/исчезновения.
 * Поддерживает ссылки на описания вложенных моделей.
 * </p>
 * <p>Позволяет для любой модели в финальном или промежуточном состоянии, а также вложенных моделей:
 * <ul>
 * <li>вычислить достижимость/разрешённость полей</li>
 * <li>проставить дефолтные значения в операциях добавления/редактирования сущности</li>
 * <li>осуществить сброс значений "исчёзнувших" полей модели</li>
 * </ul>
 * </p>
 */
public final class CoreModel<M extends ModelWithId> {
    private final Class<M> modelClass;
    private final Map<ModelProperty<? super M, ?>, ModelPropertyDefault<M, ?>> propertyDefaults;
    private final Map<ModelProperty<? super M, ?>, ChangePredicate<M>> conditionalProperties;

    private CoreModel(Class<M> modelClass,
                      Map<ModelProperty<? super M, ?>, ModelPropertyDefault<M, ?>> defaults,
                      Map<ModelProperty<? super M, ?>, ChangePredicate<M>> conditionalProperties) {
        this.modelClass = modelClass;
        this.propertyDefaults = defaults;
        this.conditionalProperties = conditionalProperties;
    }

    private CoreModel(Builder<M> builder) {
        this(
                builder.modelClass,
                ImmutableMap.copyOf(
                        builder.propertyDefaults.entrySet().stream()
                                // complete default builders
                                .map(entry -> Maps.immutableEntry(entry.getKey(), entry.getValue().build()))
                                .collect(Collectors.toUnmodifiableMap(
                                        Map.Entry::getKey,
                                        Map.Entry::getValue
                                ))
                ),
                ImmutableMap.copyOf(builder.conditionalProperties)
        );
    }

    public static <M extends ModelWithId> Builder<M> forClass(Class<M> modelClass) {
        return new Builder<>(modelClass);
    }

    public ModelPropertiesHolder resolveEditableProperties(M model) {
        PreparedChanges<M> preparedChanges = PreparedChanges.withoutChanges(model);
        return resolveEditableProperties(preparedChanges);
    }

    public ModelPropertiesHolder resolveEditableProperties(PreparedChanges<M> preparedChanges) {
        ModelPropertiesHolder holder = new ModelPropertiesHolder();

        conditionalProperties.forEach((property, predicate) -> {
            if (predicate.test(preparedChanges)) {
                holder.enablePath(property);
            }
        });

        return holder;
    }

    public void processAddDefaults(M model, Multimap<Model, ModelProperty<?, ?>> incomingFields) {
        var modelProperties = StreamEx.of(incomingFields.get(model)).toSet();
        ChangesRegistry<M> changesRegistry = ChangesRegistry.of(model, modelProperties);
        propertyDefaults.values()
                .forEach(def -> def.processAddDefaults(changesRegistry));

    }

    public void processEditDefaults(AppliedChanges<M> changes) {
        processEditDefaults(changes.getModel(), ChangesRegistry.of(changes));
    }

    public void processEditDefaults(M unmodifiedModel, ModelChanges<M> modelChanges) {
        ChangesRegistry<M> changesRegistry = ChangesRegistry.of(modelChanges);

        processEditDefaults(unmodifiedModel, changesRegistry);
    }

    private void processEditDefaults(M unmodifiedModel, ChangesRegistry<M> changesRegistry) {
        propertyDefaults.values()
                .forEach(def -> def.processEditDefaults(unmodifiedModel, changesRegistry));

        processGhostedFields(unmodifiedModel, changesRegistry);

    }

    public Class<M> getModelClass() {
        return modelClass;
    }

    public void processGhostedFields(M unmodifiedModel, ChangesRegistry<M> modelChanges) {
        PreparedChanges<M> preparedChanges =
                PreparedChanges.forEdit(unmodifiedModel, modelChanges);
        processGhostedFields(preparedChanges, modelChanges);
    }

    private void processGhostedFields(PreparedChanges<M> preparedChanges, ChangesRegistry<M> modelChanges) {
        conditionalProperties.forEach((property, shouldAppear) -> {
            if (!shouldAppear.test(preparedChanges)) {
                propertyDefaults.get(property).processGhosted(modelChanges);
            }
        });
    }

    public static class Builder<M extends ModelWithId> {
        private final Class<M> modelClass;
        private final Map<ModelProperty<? super M, ?>, ModelPropertyDefault.Builder<M, ?>> propertyDefaults;
        private final Map<ModelProperty<? super M, ?>, ChangePredicate<M>> conditionalProperties;
        @Nullable
        private final ChangePredicate<M> propertyEnabledPredicate;

        public Builder(Class<M> modelClass) {
            this(modelClass, new HashMap<>(), new HashMap<>(), null);
        }

        public Builder(Class<M> modelClass,
                       Map<ModelProperty<? super M, ?>, ModelPropertyDefault.Builder<M, ?>> propertyDefaults,
                       Map<ModelProperty<? super M, ?>, ChangePredicate<M>> conditionalProperties,
                       ChangePredicate<M> appearPredicate) {
            this.modelClass = modelClass;
            this.propertyDefaults = propertyDefaults;
            this.conditionalProperties = conditionalProperties;
            this.propertyEnabledPredicate = appearPredicate;
        }

        @SafeVarargs
        public final Builder<M> properties(ModelProperty<M, ?>... props) {
            for (ModelProperty<M, ?> property : props) {
                property(property);
            }
            return this;
        }

        public final Builder<M> properties(Set<ModelProperty<M, ?>> props) {
            for (ModelProperty<M, ?> property : props) {
                property(property);
            }
            return this;
        }

        public <V> Builder<M> property(ModelPropertyDefault.Builder<M, V> defaultsBuilder) {
            ModelProperty<? super M, V> property = defaultsBuilder.getProperty();

            ModelPropertyDefault.Builder<M, V> effectiveBuilder = (ModelPropertyDefault.Builder<M, V>)
                    propertyDefaults.compute(
                            property,
                            (k, v) -> v == null ? defaultsBuilder
                                    : ((ModelPropertyDefault.Builder) v).merge(defaultsBuilder)
                    );

            if (propertyEnabledPredicate != null) {
                effectiveBuilder.optionalWhen(propertyEnabledPredicate.negate());
            }

            if (propertyEnabledPredicate != null) {
                conditionalProperties.compute(
                        property,
                        (prop, predicate) -> {
                            if (predicate != null) {
                                return predicate.or(propertyEnabledPredicate);
                            }
                            return propertyEnabledPredicate;
                        }
                );
            } else {
                conditionalProperties.put(
                        property,
                        x -> true
                );
            }

            return this;
        }

        public <V> Builder<M> property(ModelProperty<? super M, V> property) {
            return property(propertyDefaults.computeIfAbsent(property, k -> forProperty(property)));
        }

        public Builder<M> dependentPropertiesWhen(
                ChangePredicate<M> appearCondition,
                Consumer<CoreModel.Builder<M>> configurer
        ) {
            configurer.accept(withDependentPropertyPredicate(appearCondition));
            return this;
        }

        private Builder<M> withDependentPropertyPredicate(ChangePredicate<M> appearPredicate) {
            if (propertyEnabledPredicate != null) {
                appearPredicate = propertyEnabledPredicate.and(appearPredicate);
            }
            return new Builder<>(
                    modelClass,
                    propertyDefaults,
                    conditionalProperties,
                    appearPredicate
            );
        }

        public CoreModel<M> build() {
            return new CoreModel<>(this);
        }
    }
}
