package ru.yandex.partner.core.props;

import java.util.function.Function;
import java.util.function.Supplier;

import javax.annotation.Nullable;

import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;

/**
 * Инициализатор значений по-умолчанию для полей,
 * описание основных сценариев инициализации
 * <a href="https://wiki.yandex-team.ru/partner/w/dev/2java/validation-to-java/#osobennostirealizaciioptional">
 * живёт здесь</a>
 */
public class ModelPropertyDefault<M extends ModelWithId, V> {
    private final ModelProperty<? super M, V> property;
    @Nullable
    private final Function<PreparedChanges<M>, V> defaultAddValue;
    @Nullable
    private final Function<PreparedChanges<M>, V> defaultDbValue;
    private final Supplier<V> whenGhostedValue;
    private final ChangePredicate<M> optionalPredicate;

    ModelPropertyDefault(
            ModelProperty<? super M, V> property,
            V defaultValue,
            ChangePredicate<M> optionalPredicate
    ) {
        this.property = property;
        this.defaultAddValue = x -> defaultValue;
        this.defaultDbValue = x -> defaultValue;
        this.whenGhostedValue = () -> null;
        this.optionalPredicate = optionalPredicate;
    }

    public ModelPropertyDefault(Builder<M, V> builder) {
        this.property = builder.property;
        this.optionalPredicate = builder.optionalPredicate;
        this.defaultAddValue = builder.defaultAddValue;
        this.defaultDbValue = builder.defaultDbValue;
        this.whenGhostedValue = builder.whenGhostedValue == null ? () -> null : builder.whenGhostedValue;
    }

    public static <M extends ModelWithId, V> Builder<M, V> forProperty(ModelProperty<? super M, V> property) {
        return new Builder<>(property);
    }

    public static <T extends ModelWithId, V> ModelPropertyDefault<T, V> optionalPropertyDefault(
            ModelProperty<? super T, V> property, V value) {
        return new ModelPropertyDefault<>(property, value, model -> true);
    }

    public static <T extends ModelWithId, V> ModelPropertyDefault<T, V> propertyDefault(
            ModelProperty<? super T, V> property, V value) {
        return new ModelPropertyDefault<>(property, value, model -> false);
    }

    public static <T extends ModelWithId, V> ModelPropertyDefault<T, V> propertyDefault(
            ModelProperty<? super T, V> property, V value, ChangePredicate<T> isOptional) {
        return new ModelPropertyDefault<T, V>(property, value, isOptional);
    }

    public ModelProperty<? super M, V> getProperty() {
        return property;
    }

    public void processAddDefaults(ChangesRegistry<M> changes) {
        var preparedChanges = PreparedChanges.forAdd(changes);

        var defaultValue = defaultAddValue == null
                ? defaultDbValue
                : defaultAddValue;

        if (changes.isPropChanged(property)) {
            if (!optionalPredicate.test(preparedChanges) && defaultAddValue == null) {
                return;
            }

            if (changes.getChangedProp(property) == null && defaultValue != null) {
                changes.process(
                        property,
                        defaultValue.apply(preparedChanges)
                );
            }
        } else {
            if (defaultValue != null) {
                changes.process(
                        property,
                        defaultValue.apply(preparedChanges)
                );
            }
        }
    }

    public void processEditDefaults(M unmodifiedModel, ChangesRegistry<M> changes) {
        if (!changes.isPropChanged(property)) {
            return;
        }

        var preparedChanges = PreparedChanges.forEdit(unmodifiedModel, changes);

        if (!optionalPredicate.test(preparedChanges)) {
            return;
        }

        if (changes.getChangedProp(property) == null && defaultDbValue != null) {
            changes.process(
                    property,
                    defaultDbValue.apply(preparedChanges)
            );
        }
    }

    public void processGhosted(ChangesRegistry<M> changes) {
        changes.process(
                property,
                whenGhostedValue.get()
        );
    }

    public static class Builder<M extends ModelWithId, V> {
        private final ModelProperty<? super M, V> property;
        private ChangePredicate<M> optionalPredicate = x -> false;
        private Function<PreparedChanges<M>, V> defaultAddValue;
        private Function<PreparedChanges<M>, V> defaultDbValue;
        private Supplier<V> whenGhostedValue;

        public Builder(ModelProperty<? super M, V> property) {
            this.property = property;
        }

        public ModelProperty<? super M, V> getProperty() {
            return property;
        }

        public Builder<M, V> optional() {
            this.optionalPredicate = x -> true;
            return this;
        }

        public Builder<M, V> optionalWhen(ChangePredicate<M> when) {
            this.optionalPredicate = when.and(this.optionalPredicate);
            return this;
        }

        public Builder<M, V> optionalWhen(ChangePredicate<M> when, V defaultValue) {
            this.optionalPredicate = when;
            return withDefaultValue(defaultValue);
        }

        public Builder<M, V> mandatoryOnAdd(Supplier<V> defaultSupplier) {
            this.optionalPredicate = PreparedChanges::isAddOperation;
            return withDefaultValueOnAdd(defaultSupplier);
        }

        public Builder<M, V> whenGhosted(V value) {
            this.whenGhostedValue = () -> value;
            return this;
        }

        public Builder<M, V> withDefaultValue(V defaultValue) {
            this.defaultDbValue = x -> defaultValue;
            return this;
        }

        public Builder<M, V> withDefaultValueOnAdd(V defaultValue) {
            this.defaultAddValue = x -> defaultValue;
            return this;
        }

        public Builder<M, V> withDefaultValueOnAdd(Supplier<V> defaultSupplier) {
            this.defaultAddValue = x -> defaultSupplier.get();
            return this;
        }

        public ModelPropertyDefault<M, V> build() {
            return new ModelPropertyDefault<>(this);
        }

        public Builder<M, V> merge(Builder<M, V> other) {
            if (other.defaultDbValue != null) {
                this.defaultDbValue = other.defaultDbValue;
            }
            if (other.defaultAddValue != null) {
                this.defaultAddValue = other.defaultAddValue;
            }
            if (other.whenGhostedValue != null) {
                this.whenGhostedValue = other.whenGhostedValue;
            }
            this.optionalWhen(other.optionalPredicate);
            return this;
        }
    }
}

