package ru.yandex.partner.core.props;

import javax.annotation.Nullable;

import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;

/**
 * Вспомогательный класс, введён для обобщения апи инициализации дефолтов, зависимых полей,
 * а также для вычисления достижимых полей.
 * <p>
 * В операции add отсутствует unmodifiedModel.
 * В операции edit есть и unmodifiedModel, и modelChanges.
 */
public final class PreparedChanges<M extends ModelWithId> {
    @Nullable
    private final M model;
    @Nullable
    private final ChangesRegistry<M> changes;
    private final boolean isAddOperation;

    private PreparedChanges(@Nullable M model, @Nullable ChangesRegistry<M> changes, boolean isAddOperation) {
        this.model = model;
        this.changes = changes;
        this.isAddOperation = isAddOperation;
    }

    public static <M extends ModelWithId> PreparedChanges<M> forEdit(M model, ChangesRegistry<M> changes) {
        return new PreparedChanges<>(model, changes, false);
    }

    public static <M extends ModelWithId> PreparedChanges<M> forAdd(ChangesRegistry<M> changes) {
        return new PreparedChanges<>(null, changes, true);
    }

    public static <M extends ModelWithId> PreparedChanges<M> withoutChanges(M model) {
        return new PreparedChanges<>(model, null, false);
    }

    /**
     * Возвращаем значение так, будто изменение уже применено.
     */
    public <V> V propertyValue(ModelProperty<? super M, V> prop) {
        if (changes != null && changes.isPropChanged(prop)) {
            return changes.getChangedProp(prop);
        }

        if (model != null) {
            return prop.get(model);
        }

        // would rather use undef...
        return null;
    }

    public boolean isAddOperation() {
        return isAddOperation;
    }
}
