package ru.yandex.partner.core.multistate.block;

import java.util.EnumSet;
import java.util.List;
import java.util.Set;

import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;

import ru.yandex.partner.core.messages.BlockStateMsg;
import ru.yandex.partner.libs.i18n.GettextMsg;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static ru.yandex.partner.core.multistate.block.BlockStateFlag.CHECK_STATISTICS;
import static ru.yandex.partner.core.multistate.block.BlockStateFlag.DELETED;
import static ru.yandex.partner.core.multistate.block.BlockStateFlag.DELETED_WITH_PAGE;
import static ru.yandex.partner.core.multistate.block.BlockStateFlag.WORKING;

public class BlockMultistateTest {

    @ParameterizedTest
    @MethodSource("testData")
    public void testMultistate(Long multistateValue, Set<BlockStateFlag> flags, List<GettextMsg> msgs) {

        BlockMultistate multistate = new BlockMultistate(multistateValue);

        assertEquals(multistateValue, multistate.toMultistateValue());

        for (BlockStateFlag flag : BlockStateFlag.values()) {
            assertEquals(flags.contains(flag), multistate.test(flag));
        }

        assertEquals(flags.contains(DELETED) && (flags.contains(WORKING) ||
                        flags.contains(CHECK_STATISTICS) || flags.contains(DELETED_WITH_PAGE)),
                multistate.test(DELETED.and(WORKING.or(CHECK_STATISTICS).or(DELETED_WITH_PAGE))));

        assertEquals(flags, Set.copyOf(multistate.getEnabledFlags()));
        assertEquals(msgs, multistate.getNameMessages());

        for (BlockStateFlag flag : BlockStateFlag.values()) {
            multistate.setFlag(flag, true);
            assertTrue(multistate.test(flag));
            multistate.setFlag(flag, false);
            assertFalse(multistate.test(flag));
        }

        assertEquals(0L, multistate.toMultistateValue());
    }

    public static List<Arguments> testData() {
        return List.of(
                Arguments.of(0L, Set.of(), List.of(BlockStateMsg.NEW)),
                Arguments.of(1L, Set.of(DELETED),
                        List.of(BlockStateMsg.ARCHIVED)),
                Arguments.of(2L, Set.of(WORKING),
                        List.of(BlockStateMsg.WORKING)),
                Arguments.of(3L, EnumSet.of(DELETED, WORKING),
                        List.of(BlockStateMsg.ARCHIVED, BlockStateMsg.WORKING)),
                Arguments.of(6L, EnumSet.of(WORKING, CHECK_STATISTICS),
                        List.of(BlockStateMsg.WORKING, BlockStateMsg.CHECK_STATISTICS))
        );
    }

}
