package ru.yandex.partner.core.multistate.user;

import java.util.EnumSet;
import java.util.List;
import java.util.Set;

import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;

import ru.yandex.partner.core.messages.UserStateMsg;
import ru.yandex.partner.libs.i18n.GettextMsg;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static ru.yandex.partner.core.multistate.user.UserStateFlag.BLOCKED;
import static ru.yandex.partner.core.multistate.user.UserStateFlag.CONTACTS_PROVIDED;
import static ru.yandex.partner.core.multistate.user.UserStateFlag.NEED_CREATE_IN_BANNER_STORE;
import static ru.yandex.partner.core.multistate.user.UserStateFlag.NEED_YAN_CONTRACT;

public class UserMultistateTest {

    @ParameterizedTest
    @MethodSource("testData")
    public void testMultistate(Long multistateValue, Set<UserStateFlag> flags, List<GettextMsg> msgs) {

        UserMultistate multistate = new UserMultistate(multistateValue);

        assertEquals(multistateValue, multistate.toMultistateValue());

        for (UserStateFlag flag : UserStateFlag.values()) {
            assertEquals(flags.contains(flag), multistate.test(flag));
        }

        assertEquals(flags.contains(CONTACTS_PROVIDED) && (flags.contains(NEED_CREATE_IN_BANNER_STORE) ||
                        flags.contains(NEED_YAN_CONTRACT) || flags.contains(BLOCKED)),
                multistate.test(CONTACTS_PROVIDED.and(NEED_CREATE_IN_BANNER_STORE.or(NEED_YAN_CONTRACT).or(BLOCKED))));

        assertEquals(flags, Set.copyOf(multistate.getEnabledFlags()));
        assertEquals(msgs, multistate.getNameMessages());

        for (UserStateFlag flag : UserStateFlag.values()) {
            multistate.setFlag(flag, true);
            assertTrue(multistate.test(flag));
            multistate.setFlag(flag, false);
            assertFalse(multistate.test(flag));
        }

        assertEquals(0L, multistate.toMultistateValue());
    }

    public static List<Arguments> testData() {
        return List.of(
                Arguments.of(0L, Set.of(), List.of(UserStateMsg.NEW)),
                Arguments.of(1L, Set.of(CONTACTS_PROVIDED),
                        List.of(UserStateMsg.CONTACTS_PROVIDED)),
                Arguments.of(2L, Set.of(NEED_CREATE_IN_BANNER_STORE),
                        List.of(UserStateMsg.NEED_CREATE_IN_BANNER_STORE)),
                Arguments.of(3L, EnumSet.of(CONTACTS_PROVIDED, NEED_CREATE_IN_BANNER_STORE),
                        List.of(UserStateMsg.CONTACTS_PROVIDED, UserStateMsg.NEED_CREATE_IN_BANNER_STORE)),
                Arguments.of(9L, EnumSet.of(CONTACTS_PROVIDED, NEED_YAN_CONTRACT),
                        List.of(UserStateMsg.CONTACTS_PROVIDED, UserStateMsg.NEED_YAN_CONTRACT))
        );
    }

}
