package ru.yandex.partner.core.props;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang3.tuple.Pair;
import org.junit.jupiter.api.Test;

import ru.yandex.direct.model.ModelChanges;
import ru.yandex.partner.core.entity.block.model.Geo;


import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;

/**
 * <a href="https://wiki.yandex-team.ru/partner/w/dev/2java/validation-to-java/#osobennostirealizaciioptional">
 * Тесты в соответствии с описанием работы дефолтов/optional
 * </a>
 */
class ModelPropertyDefaultTest {

    @Test
    void simpleOptionalField() {
        ModelPropertyDefault<Geo, BigDecimal> mincpmOptionalField =
                ModelPropertyDefault.forProperty(Geo.CPM)
                        .optional()
                        .build();

        // Редактирование - не прислали новое значение
        ModelChanges<Geo> mc = new ModelChanges<>(null, Geo.class);
        mincpmOptionalField.processEditDefaults(new Geo().withCpm(BigDecimal.TEN), ChangesRegistry.of(mc));
        assertTrue(mc.getChangedPropsNames().isEmpty());

        // Добавление - не прислали новое значение
        ModelChanges<Geo> mc1 = new ModelChanges<>(null, Geo.class);
        mincpmOptionalField.processAddDefaults(
                ChangesRegistry.of(mc1)
        );
        assertTrue(mc1.getChangedPropsNames().isEmpty());

        for (var newValue : Arrays.asList(null, BigDecimal.valueOf(12L))) {
            // Редактирование - значение/null
            ModelChanges<Geo> mc2 = new ModelChanges<>(null, Geo.class)
                    .process(newValue, Geo.CPM);
            mincpmOptionalField.processEditDefaults(
                    new Geo().withCpm(BigDecimal.TEN),
                    ChangesRegistry.of(mc2)
            );
            assertEquals(Set.of(Geo.CPM), mc2.getChangedPropsNames());
            assertEquals(newValue, mc2.getChangedProp(Geo.CPM));

            // добавление - значение/null
            ModelChanges<Geo> mc3 = new ModelChanges<>(null, Geo.class)
                    .process(newValue, Geo.CPM);
            mincpmOptionalField.processAddDefaults(
                    ChangesRegistry.of(mc3)
            );
            assertEquals(Set.of(Geo.CPM), mc3.getChangedPropsNames());
            assertEquals(newValue, mc3.getChangedProp(Geo.CPM));
        }
    }

    @Test
    void simpleRequiredField() {
        ModelPropertyDefault<Geo, BigDecimal> mincpmOptionalField =
                ModelPropertyDefault.forProperty(Geo.CPM).build();

        // Редактирование - не прислали новое значение
        ModelChanges<Geo> mc = new ModelChanges<>(null, Geo.class);
        mincpmOptionalField.processEditDefaults(new Geo().withCpm(BigDecimal.TEN), ChangesRegistry.of(mc));
        assertTrue(mc.getChangedPropsNames().isEmpty());

        // Добавление - не прислали новое значение
        ModelChanges<Geo> mc1 = new ModelChanges<>(null, Geo.class);
        mincpmOptionalField.processAddDefaults(ChangesRegistry.of(mc1));
        assertTrue(mc.getChangedPropsNames().isEmpty());

        for (var newValue : Arrays.asList(null, BigDecimal.valueOf(12L))) {
            // Редактирование - значение/null
            ModelChanges<Geo> mc2 = new ModelChanges<>(null, Geo.class)
                    .process(newValue, Geo.CPM);
            mincpmOptionalField.processEditDefaults(
                    new Geo().withCpm(BigDecimal.TEN),
                    ChangesRegistry.of(mc2)
            );
            assertEquals(Set.of(Geo.CPM), mc2.getChangedPropsNames());
            assertEquals(newValue, mc2.getChangedProp(Geo.CPM));

            // добавление - значение/null
            ModelChanges<Geo> mc3 = new ModelChanges<>(null, Geo.class)
                    .process(newValue, Geo.CPM);
            mincpmOptionalField.processAddDefaults(
                    ChangesRegistry.of(mc3)
            );
            assertEquals(Set.of(Geo.CPM), mc3.getChangedPropsNames());
            assertEquals(newValue, mc3.getChangedProp(Geo.CPM));
        }
    }

    @Test
    void simpleOptionalFieldWithDefault() {
        BigDecimal defaultValue = BigDecimal.ONE;
        ModelPropertyDefault<Geo, BigDecimal> mincpmOptionalField =
                ModelPropertyDefault.forProperty(Geo.CPM)
                        .optional()
                        .withDefaultValue(defaultValue)
                        .build();

        // Редактирование - не прислали новое значение
        ModelChanges<Geo> mc = new ModelChanges<>(null, Geo.class);
        mincpmOptionalField.processEditDefaults(new Geo().withCpm(BigDecimal.TEN), ChangesRegistry.of(mc));
        assertTrue(mc.getChangedPropsNames().isEmpty());

        // Добавление - не прислали новое значение
        ModelChanges<Geo> mc1 = new ModelChanges<>(null, Geo.class);
        mincpmOptionalField.processAddDefaults(
                ChangesRegistry.of(mc1)
        );
        assertEquals(Set.of(Geo.CPM), mc1.getChangedPropsNames());
        assertEquals(defaultValue, mc1.getChangedProp(Geo.CPM));

        for (var pair : List.<Pair<BigDecimal, BigDecimal>>of(
                Pair.of(null, defaultValue), Pair.of(BigDecimal.valueOf(12L), BigDecimal.valueOf(12L)))) {
            var processValue = pair.getLeft();
            var expectValue = pair.getRight();
            // Редактирование - значение/null
            ModelChanges<Geo> mc2 = new ModelChanges<>(null, Geo.class)
                    .process(processValue, Geo.CPM);
            mincpmOptionalField.processEditDefaults(
                    new Geo().withCpm(BigDecimal.TEN),
                    ChangesRegistry.of(mc2)
            );
            assertEquals(Set.of(Geo.CPM), mc2.getChangedPropsNames());
            assertEquals(expectValue, mc2.getChangedProp(Geo.CPM));

            // добавление - значение/null
            ModelChanges<Geo> mc3 = new ModelChanges<>(null, Geo.class)
                    .process(processValue, Geo.CPM);
            mincpmOptionalField.processAddDefaults(
                    ChangesRegistry.of(mc3)
            );
            assertEquals(Set.of(Geo.CPM), mc3.getChangedPropsNames());
            assertEquals(expectValue, mc3.getChangedProp(Geo.CPM));
        }
    }

    @Test
    void simpleRequiredFieldWithDefault() {
        BigDecimal defaultValue = BigDecimal.ONE;
        ModelPropertyDefault<Geo, BigDecimal> mincpmOptionalField =
                ModelPropertyDefault.forProperty(Geo.CPM)
                        .withDefaultValue(defaultValue).build();

        // Редактирование - не прислали новое значение
        ModelChanges<Geo> mc = new ModelChanges<>(null, Geo.class);
        mincpmOptionalField.processEditDefaults(new Geo().withCpm(BigDecimal.TEN), ChangesRegistry.of(mc));
        assertTrue(mc.getChangedPropsNames().isEmpty());

        // Добавление - не прислали новое значение
        ModelChanges<Geo> mc1 = new ModelChanges<>(null, Geo.class);
        mincpmOptionalField.processAddDefaults(ChangesRegistry.of(mc1));
        assertTrue(mc.getChangedPropsNames().isEmpty());

        for (var newValue : Arrays.asList(null, BigDecimal.valueOf(12L))) {
            // Редактирование - значение/null
            ModelChanges<Geo> mc2 = new ModelChanges<>(null, Geo.class)
                    .process(newValue, Geo.CPM);
            mincpmOptionalField.processEditDefaults(
                    new Geo().withCpm(BigDecimal.TEN),
                    ChangesRegistry.of(mc2)
            );
            assertEquals(Set.of(Geo.CPM), mc2.getChangedPropsNames());
            assertEquals(newValue, mc2.getChangedProp(Geo.CPM));

            // добавление - значение/null
            ModelChanges<Geo> mc3 = new ModelChanges<>(null, Geo.class)
                    .process(newValue, Geo.CPM);
            mincpmOptionalField.processAddDefaults(
                    ChangesRegistry.of(mc3)
            );
            assertEquals(Set.of(Geo.CPM), mc3.getChangedPropsNames());
            assertEquals(newValue, mc3.getChangedProp(Geo.CPM));
        }
    }

    @Test
    void simpleOptionalFieldWithAddDefault() {
        BigDecimal addDefaultValue = BigDecimal.ONE;
        ModelPropertyDefault<Geo, BigDecimal> mincpmOptionalField =
                ModelPropertyDefault.forProperty(Geo.CPM)
                        .optional()
                        .withDefaultValueOnAdd(addDefaultValue)
                        .build();

        // Редактирование - не прислали новое значение
        ModelChanges<Geo> mc = new ModelChanges<>(null, Geo.class);
        mincpmOptionalField.processEditDefaults(new Geo().withCpm(BigDecimal.TEN), ChangesRegistry.of(mc));
        assertTrue(mc.getChangedPropsNames().isEmpty());

        // Добавление - не прислали новое значение
        ModelChanges<Geo> mc1 = new ModelChanges<>(null, Geo.class);
        mincpmOptionalField.processAddDefaults(ChangesRegistry.of(mc1));
        assertEquals(Set.of(Geo.CPM), mc1.getChangedPropsNames());
        assertEquals(addDefaultValue, mc1.getChangedProp(Geo.CPM));

        for (var newValue : Arrays.asList(null, BigDecimal.valueOf(12L))) {
            // Редактирование - значение/null
            ModelChanges<Geo> mc2 = new ModelChanges<>(null, Geo.class)
                    .process(newValue, Geo.CPM);
            mincpmOptionalField.processEditDefaults(
                    new Geo().withCpm(BigDecimal.TEN),
                    ChangesRegistry.of(mc2)
            );
            assertEquals(Set.of(Geo.CPM), mc2.getChangedPropsNames());
            assertEquals(newValue, mc2.getChangedProp(Geo.CPM));

            // добавление - значение/null
            ModelChanges<Geo> mc3 = new ModelChanges<>(null, Geo.class)
                    .process(newValue, Geo.CPM);
            mincpmOptionalField.processAddDefaults(
                    ChangesRegistry.of(mc3)
            );
            assertEquals(Set.of(Geo.CPM), mc3.getChangedPropsNames());
            assertEquals(newValue == null ? addDefaultValue : newValue, mc3.getChangedProp(Geo.CPM));
        }
    }

    @Test
    void simpleRequiredFieldWithAddDefault() {
        BigDecimal addDefaultValue = BigDecimal.ONE;
        ModelPropertyDefault<Geo, BigDecimal> mincpmOptionalField =
                ModelPropertyDefault.forProperty(Geo.CPM)
                        .withDefaultValueOnAdd(addDefaultValue)
                        .build();

        // Редактирование - не прислали новое значение
        ModelChanges<Geo> mc = new ModelChanges<>(null, Geo.class);
        mincpmOptionalField.processEditDefaults(new Geo().withCpm(BigDecimal.TEN), ChangesRegistry.of(mc));
        assertTrue(mc.getChangedPropsNames().isEmpty());

        // Добавление - не прислали новое значение
        ModelChanges<Geo> mc1 = new ModelChanges<>(null, Geo.class);
        mincpmOptionalField.processAddDefaults(ChangesRegistry.of(mc1));
        assertTrue(mc.getChangedPropsNames().isEmpty());

        for (var newValue : Arrays.asList(null, BigDecimal.valueOf(12L))) {
            // Редактирование - значение/null
            ModelChanges<Geo> mc2 = new ModelChanges<>(null, Geo.class)
                    .process(newValue, Geo.CPM);
            mincpmOptionalField.processEditDefaults(
                    new Geo().withCpm(BigDecimal.TEN),
                    ChangesRegistry.of(mc2)
            );
            assertEquals(Set.of(Geo.CPM), mc2.getChangedPropsNames());
            assertEquals(newValue, mc2.getChangedProp(Geo.CPM));

            // добавление - значение/null
            ModelChanges<Geo> mc3 = new ModelChanges<>(null, Geo.class)
                    .process(newValue, Geo.CPM);
            mincpmOptionalField.processAddDefaults(
                    ChangesRegistry.of(mc3)
            );
            assertEquals(Set.of(Geo.CPM), mc3.getChangedPropsNames());
            assertEquals(newValue == null ? addDefaultValue : newValue, mc3.getChangedProp(Geo.CPM));
        }
    }
}
