package ru.yandex.partner.libs.extservice.balance;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.Collection;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.annotations.VisibleForTesting;

import ru.yandex.partner.libs.extservice.balance.method.partnercontract.BalancePartnerContract;
import ru.yandex.partner.libs.extservice.balance.method.partnercontract.Bank;
import ru.yandex.partner.libs.extservice.balance.method.partnercontract.Collateral;
import ru.yandex.partner.libs.extservice.balance.method.partnercontract.Contract;
import ru.yandex.partner.libs.extservice.balance.method.partnercontract.Person;

public class BalanceResponseConverter {

    private BalanceResponseConverter() {
        // just static methods
    }

    public static Person convertToPerson(Map<Object, Object> map) {
        return Optional.ofNullable(map.get("Person"))
                .map(personObject -> {
                    var personMap = (Map<Object, Object>) personObject;
                    if (personMap.isEmpty()) {
                        return null;
                    }
                    Integer id = parseToInteger(personMap.get("id"));
                    Integer clientId = parseToInteger(personMap.get("client_id"));

                    if (id == null) {
                        throw new BalanceException("Person.id is NULL");
                    }

                    if (clientId == null) {
                        throw new BalanceException("Person.client_id is NULL");
                    }

                    return new Person.Builder()
                            .withId(id.longValue())
                            .withClientId(clientId.longValue())
                            .withName((String) personMap.get("login"))
                            .withName((String) personMap.get("name"))
                            .withLongname((String) personMap.get("longname"))
                            .withPhone((String) personMap.get("phone"))
                            .withEmail((String) personMap.get("email"))
                            .withFax((String) personMap.get("fax"))
                            .withLegaladdress((String) personMap.get("legaladdress"))
                            .withRepresentative((String) personMap.get("representative"))
                            .withAccount((String) personMap.get("account"))
                            .withAuthorityDocType((String) personMap.get("authority_doc_type"))
                            .withBenBank((String) personMap.get("ben_bank"))
                            .withBenAccount(personMap.get("ben_account") == null ?
                                    "" : (String) personMap.get("ben_account"))
                            .withBik((String) personMap.get("bik"))
                            .withIban((String) personMap.get("iban"))
                            .withInn((String) personMap.get("inn"))
                            .withKpp((String) personMap.get("kpp"))
                            .withOther((String) personMap.get("other"))
                            .withPersonAccount((String) personMap.get("person_account"))
                            .withPostaddress((String) personMap.get("postaddress"))
                            .withSignerPersonName((String) personMap.get("signer_person_name"))
                            .withSignerPositionName((String) personMap.get("signer_position_name"))
                            .withSwift((String) personMap.get("swift"))
                            .withYamoneyWallet((String) personMap.get("yamoney_wallet"))
                            .withPayoneerWallet(personMap.get("payoneer_wallet") == null ?
                                    "" : (String) personMap.get("payoneer_wallet"))
                            .withType((String) personMap.get("type"))
                            .build();
                })
                .orElse(new Person.Builder().build());
    }

    private static Integer parseToInteger(Object obj) {
        if (obj == null) {
            return null;
        }

        if (obj instanceof String) {
            if (((String) obj).isEmpty()) {
                return null;
            } else {
                return Integer.parseInt((String) obj);
            }
        }
        return (Integer) obj;
    }

    @VisibleForTesting
    public static BalancePartnerContract convertPartnerContract(Object object) {
        try {
            var map = (Map<Object, Object>) object;

            Person person = convertToPerson(map);
            Contract contract = convertToContract(map);
            Collection<Collateral> collaterals = convertToCollaterals(map);
            return new BalancePartnerContract(person, contract, collaterals);
        } catch (ClassCastException | NumberFormatException | DateTimeParseException e) {
            throw new BalanceException("Unexpected types of xml parameters", e);
        }
    }

    public static Bank convertBank(Object object, String bankIdType, String bankId) {
        try {
            var map = (Map<Object, Object>) object;

            boolean active = parseToInteger(map.getOrDefault("hidden", 0)) == 0
                    && map.get("hidden_dt") == null;

            return new Bank(
                    active,
                    bankId,
                    bankIdType,
                    (String) map.get("name")
            );
        } catch (ClassCastException | NumberFormatException | DateTimeParseException e) {
            throw new BalanceException("Unexpected types of xml parameters", e);
        }
    }

    private static Contract convertToContract(Map<Object, Object> map) {
        return Optional.ofNullable(map.get("Contract"))
                .map(contractObject -> {
                    var contractMap = (Map<Object, Object>) contractObject;
                    if (contractMap.isEmpty()) {
                        return null;
                    }

                    LocalDate dtLocalDate = parseToLocalDate((String) contractMap.get("dt"));

                    if (dtLocalDate == null) {
                        throw new BalanceException("Contract.dt is NULL");
                    }

                    return new Contract.Builder()
                            .withDt(dtLocalDate)
                            .withType((String) contractMap.get("type"))
                            .withExternalId((String) contractMap.get("external_id"))
                            .withContractType(parseToInteger(contractMap.get("contract_type")))
                            .withRewardType(parseToInteger(contractMap.get("reward_type")))
                            .withEndDt(parseToLocalDate((String) contractMap.get("end_dt")))
                            .withIsCancelled(parseToLocalDate((String) contractMap.get("is_cancelled")))
                            .withIsSigned(parseToLocalDate((String) contractMap.get("is_signed")))
                            .withIsFaxed(parseToLocalDate((String) contractMap.get("is_faxed")))
                            .withTestMode(parseToInteger(contractMap.get("test_mode")))
                            .withCurrency(parseToInteger(contractMap.get("currency")))
                            .withVat((String) contractMap.get("vat"))
                            .withPayTo(parseToInteger(contractMap.get("pay_to")))
                            .withPersonId(
                                    parseToNumber(contractMap.get("person_id"))
                                            .map(Number::longValue)
                                            .orElse(null)
                            )
                            .withFirm(parseToInteger(contractMap.get("firm")))
                            .withContract2Id(parseToInteger(contractMap.get("contract2_id")))
                            .withNds(parseToInteger(contractMap.get("nds")))
                            .withTestMode(parseToInteger(contractMap.get("test_mode")))
                            .build();
                })
                .orElse(null);
    }

    private static Collection<Collateral> convertToCollaterals(Map<Object, Object> map) {
        return Optional.ofNullable(map.get("Collaterals"))
                .stream()
                .flatMap(collateralObjects -> Stream.of((Object[]) collateralObjects))
                .map(BalanceResponseConverter::convertToCollateral)
                .collect(Collectors.toList());
    }

    private static Collateral convertToCollateral(Object object) {
        Map<Object, Object> collateralMap = (Map<Object, Object>) object;
        Integer collateralTypeId = parseToInteger(collateralMap.get("collateral_type_id"));
        LocalDate dtLocalDate = parseToLocalDate((String) collateralMap.get("dt"));

        if (collateralTypeId == null) {
            throw new BalanceException("Collateral.collateral_type_id is NULL");
        }
        if (dtLocalDate == null) {
            throw new BalanceException("Collateral.dt is NULL");
        }

        return new Collateral.Builder()
                .withCollateralTypeId(collateralTypeId)
                .withBalanceClassType((String) collateralMap.get("class"))
                .withDt(dtLocalDate)
                .withEndDt(parseToLocalDate((String) collateralMap.get("end_dt")))
                .withIsCancelled(parseToLocalDate((String) collateralMap.get("is_cancelled")))
                .withIsSigned(parseToLocalDate((String) collateralMap.get("is_signed")))
                .withIsFaxed(parseToLocalDate((String) collateralMap.get("is_faxed")))
                .withNds(parseToInteger(collateralMap.get("nds")))
                .withPartnerPct(parseToBigDecimal((String) collateralMap.get("partner_pct")))
                .withAgregatorPct(parseToBigDecimal((String) collateralMap.get("agregator_pct")))
                .withMkbPrice((String) collateralMap.get("mrk_price"))
                .withSearchForms(parseToInteger(collateralMap.get("search_forms")))
                .withEndReason(parseToInteger(collateralMap.get("end_reason")))
                .withPayTo(parseToInteger(collateralMap.get("pay_to")))
                .withTailTime(parseToInteger(collateralMap.get("tail_time")))
                .withDistributionPlaces((String) collateralMap.get("distribution_places"))
                .withProductSearch((String) collateralMap.get("product_search"))
                .withProductSearchf((String) collateralMap.get("product_searchf"))
                .withProductOptions((String) collateralMap.get("product_options"))
                .withProductsDownload((String) collateralMap.get("products_download"))
                .withDownloadDomains((String) collateralMap.get("download_domains"))
                .withInstallPrice(parseToBigDecimal((String) collateralMap.get("install_price")))
                .withInstallSoft((String) collateralMap.get("install_soft"))
                .withRewardType(parseToInteger(collateralMap.get("reward_type")))
                .build();
    }

    private static LocalDate parseToLocalDate(String str) {
        if (str == null || str.isEmpty()) {
            return null;
        }

        return LocalDate.parse(str, DateTimeFormatter.ISO_DATE);
    }

    private static BigDecimal parseToBigDecimal(String str) {
        if (str == null || str.isEmpty()) {
            return null;
        }

        return new BigDecimal(str);
    }

    private static Optional<Number> parseToNumber(Object object) {
        if (object == null) {
            return Optional.empty();
        } else if (object instanceof String) {
            return ((String) object).isEmpty()
                    ? Optional.empty()
                    : Optional.of(new BigDecimal((String) object));
        } else {
            return Optional.of((Number) object);
        }
    }
}
