package ru.yandex.partner.libs.extservice.balance;

import java.time.Duration;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.primitives.Ints;
import org.apache.ws.commons.util.NamespaceContextImpl;
import org.apache.xmlrpc.XmlRpcException;
import org.apache.xmlrpc.client.XmlRpcClient;
import org.apache.xmlrpc.client.XmlRpcClientConfigImpl;
import org.apache.xmlrpc.client.XmlRpcHttpTransportException;
import org.apache.xmlrpc.common.TypeFactoryImpl;
import org.apache.xmlrpc.common.XmlRpcController;
import org.apache.xmlrpc.common.XmlRpcLoadException;
import org.apache.xmlrpc.common.XmlRpcStreamConfig;
import org.apache.xmlrpc.parser.NullParser;
import org.apache.xmlrpc.parser.TypeParser;
import org.apache.xmlrpc.serializer.NullSerializer;
import org.apache.xmlrpc.serializer.TypeSerializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.SAXException;

import ru.yandex.partner.libs.extservice.balance.exception.BalanceClientException;

/**
 * Базовый клиент для XmlRpc API Баланса. Используется через хелперы в других классах пакета
 * <p>
 * ВАЖНО: XML-RPC API Баланса иногда нарушает спецификацию (http://xmlrpc.scripting.com/spec.html) а именно:
 * "The body of the response is a single XML structure, a <methodResponse>,
 * which can contain a single <params> which contains a single <param> which contains a single <value>."
 * <p>
 * Нарушение заключается в том, что возвращаемое значение (value) может быть не одно.
 * Текущая используемая реализация клиента из-за этого возвращает в качестве ответа только ПОСЛЕДНЕЕ <value>
 * в <param>, а все предыдущие отбрасывает.
 * <p>
 * Пока это никак не влияет на получаемые нами значения, так как отбрасываются только константные статусы,
 * а в случае возврата списка значений в Балансе используются массивы, но это нужно учитывать при разработке.
 * Сейчас основной маркер того, что возвращаемое значение может быть отброшено - слово "кортеж" в документации
 * к АПИ.
 * Например:
 * - в ручке Balance.CreateClient (https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createclient) по документации
 * получаем кортеж из статуса, сообщения и ClientID. Значит ответ нужно обрабатывать так,
 * как будто мы получаем только ClientID.
 * - в ручке Balance.GetFirmCountryCurrency (https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getfirmcountrycurrency)
 * по документации получаем список из статуса, сообщения и данных. Значит обрабатывать нужно все три элемента
 * списка
 * <p>
 * В сервисе Balance2 таких проблем пока замечено не было.
 * <p>
 * Тип {@link Long} не поддерживается на стороне баланса библиотекой. Поля с такой размерностью передаются строками.
 */
@ParametersAreNonnullByDefault
public class BalanceXmlRpcClient {
    private static final Logger BALANCE_CALLS_LOGGER = LoggerFactory.getLogger(BalanceXmlRpcClient.class);

    private final XmlRpcClientConfigImpl clientConfig;
    private final BalanceXmlRpcConfig config;
    private final XmlRpcClient client;
    private final int retriesNum;

    public BalanceXmlRpcClient(BalanceXmlRpcConfig config) {
        this.config = config;
        clientConfig = getXmlRpcClientConfig(config);
        client = new XmlRpcClient();
        client.setTypeFactory(new XmlRpcTypeNil(client));
        client.setConfig(clientConfig);

        retriesNum = config.getRetries();
    }

    /**
     * Создать инстанс настроек XML-RPC клиента из настроек клиента Баланса
     */
    static XmlRpcClientConfigImpl getXmlRpcClientConfig(BalanceXmlRpcConfig config) {
        XmlRpcClientConfigImpl xmlRpcClientConfig = new XmlRpcClientConfigImpl();
        xmlRpcClientConfig.setServerURL(config.getServerURL());
        xmlRpcClientConfig.setConnectionTimeout(Ints.saturatedCast(config.getConnectionTimeout().toMillis()));
        xmlRpcClientConfig.setReplyTimeout(Ints.saturatedCast(config.getTimeout().toMillis()));
        return xmlRpcClientConfig;
    }

    /**
     * Выполнить серию обращений к методу XML-RPC Баланса с заданным таимаутом на одно обращение и максимальным
     * количеством повторов. Вернуть результат первого успешного обращения.
     * <p>
     * Если таимаут или максимальное количество попыток не заданы, берутся значения по-умолчанию
     */
    public Object retriedCall(String methodName, Object[] params, @Nullable Duration overrideTimeout,
                              @Nullable Integer overrideRetries) {
        int maxRetries = overrideRetries == null ? retriesNum : overrideRetries;
        for (int currentRetry = 1; ; currentRetry++) {
            try {
                // успешные запросы считаем выше по стеку
                // так как ошибка может быть упакована в тело ответа
                return rawCall(methodName, params, overrideTimeout);
            } catch (XmlRpcLoadException e) {
                String msg = "%s call with params %s error on try %s: %s"
                        .formatted(methodName, params, currentRetry, e.getMessage());
                if (currentRetry < maxRetries) {
                    BALANCE_CALLS_LOGGER.warn(msg, e);
                } else {
                    BALANCE_CALLS_LOGGER.error(msg, e);
                    throw new BalanceClientException(msg, e);
                }
            } catch (XmlRpcHttpTransportException e) {
                String msg = "%s call with params %s HTTP error on try %s: %s %s"
                        .formatted(methodName, params, currentRetry, e.getStatusCode(), e.getStatusMessage());
                if (currentRetry < maxRetries) {
                    BALANCE_CALLS_LOGGER.warn(msg, e);
                } else {
                    BALANCE_CALLS_LOGGER.error(msg, e);
                    throw new BalanceClientException(msg, e);
                }
            } catch (XmlRpcException e) {
                String msg = "%s call with params %s fault on try %s: %s"
                        .formatted(methodName, params, currentRetry, e.getMessage());
                if (currentRetry < maxRetries) {
                    BALANCE_CALLS_LOGGER.warn(msg, e);
                } else {
                    BALANCE_CALLS_LOGGER.error(msg, e);
                    throw new BalanceClientException(msg, e);
                }
            }
        }
    }

    /**
     * Выполнить непосредственное обращение к методу XML-RPC Баланса с заданным таимаутом и вернуть ответ
     * <p>
     * Если таимаут не задан, берется таимаут по-умолчанию
     */
    Object rawCall(String methodName, Object[] params, @Nullable Duration timeout) throws XmlRpcException {
        Object response;
        if (timeout == null || clientConfig.getReplyTimeout() == timeout.toMillis()) {
            response = client.execute(methodName, params);
        } else {
            XmlRpcClientConfigImpl newConfig = getXmlRpcClientConfig(config);
            newConfig.setReplyTimeout(Ints.saturatedCast(timeout.toMillis()));
            response = client.execute(newConfig, methodName, params);
        }
        return response;
    }

    /**
     * Workaround for
     * "Failed to parse server's response: Unknown type: nil"
     */
    class XmlRpcTypeNil extends TypeFactoryImpl {

        XmlRpcTypeNil(XmlRpcController pController) {
            super(pController);
        }

        public TypeParser getParser(XmlRpcStreamConfig pConfig, NamespaceContextImpl pContext, String pURI,
                                    String pLocalName) {
            if (NullSerializer.NIL_TAG.equals(pLocalName) || NullSerializer.EX_NIL_TAG.equals(pLocalName)) {
                return new NullParser();
            } else {
                return super.getParser(pConfig, pContext, pURI, pLocalName);
            }
        }

        public TypeSerializer getSerializer(XmlRpcStreamConfig pConfig, Object pObject) throws SAXException {
            if (pObject instanceof XmlRpcTypeNil) {
                return new NullSerializer();
            } else {
                return super.getSerializer(pConfig, pObject);
            }
        }
    }
}
