package ru.yandex.partner.libs.extservice.balance.method.partnercontract;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.Objects;

import ru.yandex.partner.core.CoreConstants;

public class Collateral {
    private final Integer collateralTypeId;
    private final String balanceClassType;
    private final LocalDate dt;
    private final LocalDate endDt;
    private final LocalDate isCancelled;
    private final LocalDate isSigned;
    private final LocalDate isFaxed;

    private final Integer nds;
    private final BigDecimal partnerPct;
    private final BigDecimal agregatorPct;

    // TODO {str: int}
    private final String mkbPrice;
    private final Integer searchForms;
    private final Integer endReason;
    private final Integer payTo;
    private final Integer tailTime;

    // TODO String???
    private final String distributionPlaces;
    private final String productSearch;
    private final String productSearchf;
    private final String productOptions;

    // TODO {str: money}
    private final String productsDownload;
    private final String downloadDomains;
    private final BigDecimal installPrice;
    private final String installSoft;
    private final Integer rewardType;

    public Collateral(Builder builder) {
        this.collateralTypeId = Objects.requireNonNull(builder.collateralTypeId);
        this.dt = Objects.requireNonNull(builder.dt);
        this.balanceClassType = builder.balanceClassType;
        this.endDt = builder.endDt;
        this.isCancelled = builder.isCancelled;
        this.isSigned = builder.isSigned;
        this.isFaxed = builder.isFaxed;
        this.nds = builder.nds;
        this.partnerPct = builder.partnerPct;
        this.agregatorPct = builder.agregatorPct;
        this.mkbPrice = builder.mkbPrice;
        this.searchForms = builder.searchForms;
        this.endReason = builder.endReason;
        this.payTo = builder.payTo;
        this.tailTime = builder.tailTime;
        this.distributionPlaces = builder.distributionPlaces;
        this.productSearch = builder.productSearch;
        this.productSearchf = builder.productSearchf;
        this.productOptions = builder.productOptions;
        this.productsDownload = builder.productsDownload;
        this.downloadDomains = builder.downloadDomains;
        this.installPrice = builder.installPrice;
        this.installSoft = builder.installSoft;
        this.rewardType = builder.rewardType;
    }

    public Integer getCollateralTypeId() {
        return collateralTypeId;
    }

    public LocalDate getDt() {
        return dt;
    }

    public boolean isLive(LocalDate date) {
        return
                !(
                        (date.isBefore(dt))                        // Время действия ещё не наступило
                                || (isCancelled != null)                          // Аннулировано
                                || (
                                CoreConstants.BalanceClassType
                                        .COLLATERAL.equals(balanceClassType)     // Допник должен быть подписан
                                        && !(
                                        isSigned != null                         // оригинал
                                                || isFaxed != null               // или по факсу
                                )
                                        && collateralTypeId != 2050              // и это не допник о закрытии
                        )
                );
    }

    /**
     * Проверка на "расторжение договора"
     * (collateral_type_id = 2050)
     * Это допсоглашение действует только в том случае, если пользователь
     * подписал это допсоглашение, т.е. если заполнено поле is_faxed или is_signed)
     *
     * @param date дата относительно которой проверяем
     * @return true - если это "расторжение договора" иначе false
     */
    public boolean isTermination(LocalDate date) {
        return collateralTypeId == 2050
                && isCancelled == null
                && endDt != null
                && date.isAfter(endDt)
                && (isFaxed != null || isSigned != null);
    }

    /**
     * Проверка на "уведомление о закрытии"
     * (collateral_type_id = 2090)
     * Это уведомление не нужно подписывать, оно начининает действовать с даты 'dt'
     *
     * @param date дата относительно которой проверяем
     * @return true - если это "уведомление о закрытии" иначе false
     */
    public boolean isNotificationAboutClose(LocalDate date) {
        return collateralTypeId == 2090
                && isCancelled == null
                && endDt != null
                && date.isAfter(endDt);
    }

    public LocalDate getEndDt() {
        return this.endDt;
    }

    public Integer getNds() {
        return this.nds;
    }

    public BigDecimal getPartnerPct() {
        return this.partnerPct;
    }

    public BigDecimal getAgregatorPct() {
        return this.agregatorPct;
    }

    public String getMkbPrice() {
        return this.mkbPrice;
    }

    public Integer getSearchForms() {
        return this.searchForms;
    }

    public Integer getEndReason() {
        return this.endReason;
    }

    public Integer getPayTo() {
        return this.payTo;
    }

    public Integer getTailTime() {
        return this.tailTime;
    }

    public String getDistributionPlaces() {
        return this.distributionPlaces;
    }

    public String getProductSearch() {
        return this.productSearch;
    }

    public String getProductSearchf() {
        return this.productSearchf;
    }

    public String getProductOptions() {
        return this.productOptions;
    }

    public String getProductsDownload() {
        return this.productsDownload;
    }

    public String getDownloadDomains() {
        return this.downloadDomains;
    }

    public BigDecimal getInstallPrice() {
        return this.installPrice;
    }

    public String getInstallSoft() {
        return this.installSoft;
    }

    public Integer getRewardType() {
        return this.rewardType;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof Collateral)) {
            return false;
        }
        Collateral that = (Collateral) o;
        return Objects.equals(collateralTypeId, that.collateralTypeId) &&
                Objects.equals(balanceClassType, that.balanceClassType) &&
                Objects.equals(dt, that.dt) &&
                Objects.equals(endDt, that.endDt) &&
                Objects.equals(isCancelled, that.isCancelled) &&
                Objects.equals(isSigned, that.isSigned) &&
                Objects.equals(isFaxed, that.isFaxed) &&
                Objects.equals(nds, that.nds) &&
                Objects.equals(partnerPct, that.partnerPct) &&
                Objects.equals(agregatorPct, that.agregatorPct) &&
                Objects.equals(mkbPrice, that.mkbPrice) &&
                Objects.equals(searchForms, that.searchForms) &&
                Objects.equals(endReason, that.endReason) &&
                Objects.equals(payTo, that.payTo) &&
                Objects.equals(tailTime, that.tailTime) &&
                Objects.equals(distributionPlaces, that.distributionPlaces) &&
                Objects.equals(productSearch, that.productSearch) &&
                Objects.equals(productSearchf, that.productSearchf) &&
                Objects.equals(productOptions, that.productOptions) &&
                Objects.equals(productsDownload, that.productsDownload) &&
                Objects.equals(downloadDomains, that.downloadDomains) &&
                Objects.equals(installPrice, that.installPrice) &&
                Objects.equals(installSoft, that.installSoft) &&
                Objects.equals(rewardType, that.rewardType);
    }

    @Override
    public int hashCode() {
        return Objects.hash(collateralTypeId, balanceClassType, dt, endDt, isCancelled, isSigned, isFaxed, nds,
                partnerPct, agregatorPct, mkbPrice, searchForms, endReason, payTo, tailTime, distributionPlaces,
                productSearch, productSearchf, productOptions, productsDownload, downloadDomains, installPrice,
                installSoft, rewardType);
    }

    public static class Builder {
        private Integer collateralTypeId;
        private String balanceClassType;
        private LocalDate dt;
        private LocalDate endDt;
        private LocalDate isCancelled;
        private LocalDate isSigned;
        private LocalDate isFaxed;
        private Integer nds;
        private BigDecimal partnerPct;
        private BigDecimal agregatorPct;
        private String mkbPrice;
        private Integer searchForms;
        private Integer endReason;
        private Integer payTo;
        private Integer tailTime;
        private String distributionPlaces;
        private String productSearch;
        private String productSearchf;
        private String productOptions;
        private String productsDownload;
        private String downloadDomains;
        private BigDecimal installPrice;
        private String installSoft;
        private Integer rewardType;

        public Builder withCollateralTypeId(Integer collateralTypeId) {
            this.collateralTypeId = collateralTypeId;
            return this;
        }

        public Builder withBalanceClassType(String balanceClassType) {
            this.balanceClassType = balanceClassType;
            return this;
        }

        public Builder withDt(LocalDate dt) {
            this.dt = dt;
            return this;
        }

        public Builder withEndDt(LocalDate endDt) {
            this.endDt = endDt;
            return this;
        }

        public Builder withIsCancelled(LocalDate isCancelled) {
            this.isCancelled = isCancelled;
            return this;
        }

        public Builder withIsSigned(LocalDate isSigned) {
            this.isSigned = isSigned;
            return this;
        }

        public Builder withIsFaxed(LocalDate isFaxed) {
            this.isFaxed = isFaxed;
            return this;
        }

        public Builder withNds(Integer nds) {
            this.nds = nds;
            return this;
        }

        public Builder withPartnerPct(BigDecimal partnerPct) {
            this.partnerPct = partnerPct;
            return this;
        }

        public Builder withAgregatorPct(BigDecimal agregatorPct) {
            this.agregatorPct = agregatorPct;
            return this;
        }

        public Builder withMkbPrice(String mkbPrice) {
            this.mkbPrice = mkbPrice;
            return this;
        }

        public Builder withSearchForms(Integer searchForms) {
            this.searchForms = searchForms;
            return this;
        }

        public Builder withEndReason(Integer endReason) {
            this.endReason = endReason;
            return this;
        }

        public Builder withPayTo(Integer payTo) {
            this.payTo = payTo;
            return this;
        }

        public Builder withTailTime(Integer tailTime) {
            this.tailTime = tailTime;
            return this;
        }

        public Builder withDistributionPlaces(String distributionPlaces) {
            this.distributionPlaces = distributionPlaces;
            return this;
        }

        public Builder withProductSearch(String productSearch) {
            this.productSearch = productSearch;
            return this;
        }

        public Builder withProductSearchf(String productSearchf) {
            this.productSearchf = productSearchf;
            return this;
        }

        public Builder withProductOptions(String productOptions) {
            this.productOptions = productOptions;
            return this;
        }

        public Builder withProductsDownload(String productsDownload) {
            this.productsDownload = productsDownload;
            return this;
        }

        public Builder withDownloadDomains(String downloadDomains) {
            this.downloadDomains = downloadDomains;
            return this;
        }

        public Builder withInstallPrice(BigDecimal installPrice) {
            this.installPrice = installPrice;
            return this;
        }

        public Builder withInstallSoft(String installSoft) {
            this.installSoft = installSoft;
            return this;
        }

        public Builder withRewardType(Integer rewardType) {
            this.rewardType = rewardType;
            return this;
        }

        public Collateral build() {
            return new Collateral(this);
        }
    }
}
