package ru.yandex.partner.libs.extservice.balance.method.partnercontract;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Objects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.support.MessageSourceAccessor;

import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.core.entity.balance.model.BalanceContract;
import ru.yandex.partner.libs.extservice.balance.BalanceContractUtils;
import ru.yandex.partner.libs.extservice.balance.ContractLiveData;
import ru.yandex.partner.libs.extservice.balance.message.ModelDocumentsMsg;
import ru.yandex.partner.libs.extservice.balance.message.ModelOebsMsg;

public class Contract implements ContractLiveData {
    private static final Logger LOGGER = LoggerFactory.getLogger(Contract.class);
    private final String externalId;
    private final String type;
    private final Integer contractType;
    private final Integer rewardType;
    private final LocalDate dt;
    private final LocalDate endDt;
    private final LocalDate isCancelled;
    private final LocalDate isSigned;
    private final LocalDate isFaxed;
    private final Integer testMode;
    private final Integer currency;
    private final String vat;
    private final Integer payTo;
    private final Long personId;
    private final Integer firm;
    private final Integer contract2Id;
    private final Integer nds;

    private final OverrideField<LocalDate> collateralEndDt;
    private final OverrideField<Integer> collateralRewardType;
    private final OverrideField<Integer> collateralPayTo;

    private Integer collateralNds;
    private BigDecimal collateralPartnerPct;
    private BigDecimal collateralAgregatorPct;
    private String collateralMkbPrice;
    private Integer collateralSearchForms;
    private Integer collateralEndReason;
    private Integer collateralTailTime;
    private String collateralDistributionPlaces;
    private String collateralProductSearch;
    private String collateralProductSearchf;
    private String collateralProductOptions;
    private String collateralProductsDownload;
    private String collateralDownloadDomains;
    private BigDecimal collateralInstallPrice;
    private String collateralInstallSoft;


    private Contract(Builder builder) {
        this.dt = Objects.requireNonNull(builder.dt);
        this.type = builder.type;
        this.externalId = builder.externalId;
        this.contractType = builder.contractType;
        this.rewardType = builder.rewardType;
        this.endDt = builder.endDt;
        this.isCancelled = builder.isCancelled;
        this.isSigned = builder.isSigned;
        this.isFaxed = builder.isFaxed;
        this.testMode = builder.testMode;
        this.currency = builder.currency;
        this.vat = builder.vat;
        this.payTo = builder.payTo;
        this.personId = builder.personId;
        this.firm = builder.firm;
        this.contract2Id = builder.contract2Id;
        this.nds = builder.nds;

        collateralEndDt = new OverrideField<>(endDt);
        collateralPayTo = new OverrideField<>(payTo);
        collateralRewardType = new OverrideField<>(rewardType);
    }

    @Override
    public Integer getContractType() {
        return contractType;
    }

    @Override
    public Integer getTestMode() {
        return testMode;
    }

    @Override
    public LocalDate getDt() {
        return dt;
    }

    @Override
    public LocalDate getIsSigned() {
        return isSigned;
    }

    @Override
    public LocalDate getIsFaxed() {
        return isFaxed;
    }

    @Override
    public LocalDate getIsCanceled() {
        return isCancelled;
    }

    @Override
    public LocalDate getContractEndDt() {
        return endDt;
    }

    public String getType() {
        return type;
    }

    public Integer getFirm() {
        return firm;
    }

    public Integer getContract2Id() {
        return contract2Id;
    }

    public Integer getNds() {
        return nds;
    }

    public String getContractStatus(MessageSourceAccessor messageSourceAccessor) {
        Objects.requireNonNull(messageSourceAccessor);
        if (Objects.equals(contractType, CoreConstants.BalanceContractType.OFFER)) {
            if (BalanceContractUtils.isTestMode(testMode)) {
                return messageSourceAccessor.getMessage(ModelDocumentsMsg.OFFER_TEST_MODE);
            } else if (isSigned != null) {
                return messageSourceAccessor.getMessage(ModelDocumentsMsg.OFFER_ACCEPTED);
            } else {
                LOGGER.warn("Unknown contract status");
                return "";
            }
        } else {
            if (!isSignedOrFaxed()) {
                return messageSourceAccessor.getMessage(ModelOebsMsg.SENT_BY_EMAIL);
            } else if (!isSigned()) {
                return messageSourceAccessor.getMessage(ModelOebsMsg.GOT_COPY_SEND_ORIGINAL);
            } else {
                String signed = messageSourceAccessor.getMessage(ModelOebsMsg.SIGNED);
                if (Objects.equals(contractType, CoreConstants.BalanceContractType.NEW_OFFER)) {
                    return signed;
                } else {
                    return signed.concat(" ").concat(isSigned.format(DateTimeFormatter.ofPattern("dd.MM.yyyy")));
                }
            }
        }
    }

    public boolean isSignedOrFaxed() {
        return isFaxed != null || isSigned();
    }

    public boolean isSigned() {
        return isSigned != null;
    }


    /**
     * Проверка, что договор действует
     * - начал действовать
     * - не закрыт
     * - не аннулирован
     * - подписан или отправлен по факсу
     * или договор должен быть офертой в тестовом режиме - так работают партнеры,
     * которые зарегистрировались в ПИ, но еще не оставили свои реквизиты
     *
     * @param date дата относительно которой проверяем
     * @return true если активен, иначе false
     */
    public boolean isLive(LocalDate date) {
        return BalanceContractUtils.isLiveContract(date, this);
    }

    public boolean isOferta2TestMode() {
        return BalanceContractUtils.isOferta2TestMode(contractType, testMode);
    }

    public boolean allowsToFillPart2() {
        return isOferta2TestMode() && personId == null;
    }

    public BalanceContract toBalanceContract() {
        return new BalanceContract()
                .withExternalId(externalId)
                .withDt(dt)
                .withIsSigned(isSigned)
                .withIsFaxed(isFaxed)
                .withCurrency(currency)
                .withVat(vat)
                .withRewardType(rewardType)
                .withContractType(contractType)
                .withFirm(firm)
                .withPersonId(personId)
                .withContract2Id(contract2Id)
                .withNds(nds)
                .withTestMode(testMode);
    }


    /*
     ******************************
     * OVERRIDABLE FIELDS GETTERS *
     ******************************
     */
    public LocalDate getEndDt() {
        return collateralEndDt.getValue();
    }

    public Integer getPayTo() {
        return collateralPayTo.getValue();
    }

    public Integer getRewardType() {
        return collateralRewardType.getValue();
    }

    public BigDecimal getPartnerPct() {
        return collateralPartnerPct;
    }

    public BigDecimal getAgregatorPct() {
        return collateralAgregatorPct;
    }

    public String getMkbPrice() {
        return collateralMkbPrice;
    }

    public Integer getSearchForms() {
        return collateralSearchForms;
    }

    public Integer getEndReason() {
        return collateralEndReason;
    }

    public Integer getTailTime() {
        return collateralTailTime;
    }

    public String getDistributionPlaces() {
        return collateralDistributionPlaces;
    }

    public String getProductSearch() {
        return collateralProductSearch;
    }

    public String getProductSearchf() {
        return collateralProductSearchf;
    }

    public String getProductOptions() {
        return collateralProductOptions;
    }

    public String getProductsDownload() {
        return collateralProductsDownload;
    }

    public String getDownloadDomains() {
        return collateralDownloadDomains;
    }

    public BigDecimal getInstallPrice() {
        return collateralInstallPrice;
    }

    public String getInstallSoft() {
        return collateralInstallSoft;
    }

    /*
     ******************************
     * OVERRIDABLE FIELDS SETTERS *
     ******************************
     */
    public void setCollateralEndDt(LocalDate collateralEndDt) {
        this.collateralEndDt.setValue(collateralEndDt);
    }

    public void setCollateralRewardType(Integer collateralRewardType) {
        this.collateralRewardType.setValue(collateralRewardType);
    }

    public void setCollateralPayTo(Integer collateralPayTo) {
        this.collateralPayTo.setValue(collateralPayTo);
    }

    public void setCollateralNds(Integer collateralNds) {
        this.collateralNds = collateralNds;
    }

    public void setCollateralPartnerPct(BigDecimal collateralPartnerPct) {
        this.collateralPartnerPct = collateralPartnerPct;
    }

    public void setCollateralAgregatorPct(BigDecimal collateralAgregatorPct) {
        this.collateralAgregatorPct = collateralAgregatorPct;
    }

    public void setCollateralMkbPrice(String collateralMkbPrice) {
        this.collateralMkbPrice = collateralMkbPrice;
    }

    public void setCollateralSearchForms(Integer collateralSearchForms) {
        this.collateralSearchForms = collateralSearchForms;
    }

    public void setCollateralEndReason(Integer collateralEndReason) {
        this.collateralEndReason = collateralEndReason;
    }

    public void setCollateralTailTime(Integer collateralTailTime) {
        this.collateralTailTime = collateralTailTime;
    }

    public void setCollateralDistributionPlaces(String collateralDistributionPlaces) {
        this.collateralDistributionPlaces = collateralDistributionPlaces;
    }

    public void setCollateralProductSearch(String collateralProductSearch) {
        this.collateralProductSearch = collateralProductSearch;
    }

    public void setCollateralProductSearchf(String collateralProductSearchf) {
        this.collateralProductSearchf = collateralProductSearchf;
    }

    public void setCollateralProductOptions(String collateralProductOptions) {
        this.collateralProductOptions = collateralProductOptions;
    }

    public void setCollateralProductsDownload(String collateralProductsDownload) {
        this.collateralProductsDownload = collateralProductsDownload;
    }

    public void setCollateralDownloadDomains(String collateralDownloadDomains) {
        this.collateralDownloadDomains = collateralDownloadDomains;
    }

    public void setCollateralInstallPrice(BigDecimal collateralInstallPrice) {
        this.collateralInstallPrice = collateralInstallPrice;
    }

    public void setCollateralInstallSoft(String collateralInstallSoft) {
        this.collateralInstallSoft = collateralInstallSoft;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        Contract contract = (Contract) o;
        return Objects.equals(externalId, contract.externalId) &&
                Objects.equals(type, contract.type) &&
                Objects.equals(contractType, contract.contractType) &&
                Objects.equals(rewardType, contract.rewardType) &&
                Objects.equals(dt, contract.dt) &&
                Objects.equals(endDt, contract.endDt) &&
                Objects.equals(isCancelled, contract.isCancelled) &&
                Objects.equals(isSigned, contract.isSigned) &&
                Objects.equals(isFaxed, contract.isFaxed) &&
                Objects.equals(testMode, contract.testMode) &&
                Objects.equals(currency, contract.currency) &&
                Objects.equals(vat, contract.vat) &&
                Objects.equals(payTo, contract.payTo) &&
                Objects.equals(personId, contract.personId) &&
                Objects.equals(firm, contract.firm) &&
                Objects.equals(contract2Id, contract.contract2Id) &&
                Objects.equals(nds, contract.nds) &&
                Objects.equals(collateralEndDt, contract.collateralEndDt) &&
                Objects.equals(collateralRewardType, contract.collateralRewardType) &&
                Objects.equals(collateralPayTo, contract.collateralPayTo) &&
                Objects.equals(collateralNds, contract.collateralNds) &&
                Objects.equals(collateralPartnerPct, contract.collateralPartnerPct) &&
                Objects.equals(collateralAgregatorPct, contract.collateralAgregatorPct) &&
                Objects.equals(collateralMkbPrice, contract.collateralMkbPrice) &&
                Objects.equals(collateralSearchForms, contract.collateralSearchForms) &&
                Objects.equals(collateralEndReason, contract.collateralEndReason) &&
                Objects.equals(collateralTailTime, contract.collateralTailTime) &&
                Objects.equals(collateralDistributionPlaces, contract.collateralDistributionPlaces) &&
                Objects.equals(collateralProductSearch, contract.collateralProductSearch) &&
                Objects.equals(collateralProductSearchf, contract.collateralProductSearchf) &&
                Objects.equals(collateralProductOptions, contract.collateralProductOptions) &&
                Objects.equals(collateralProductsDownload, contract.collateralProductsDownload) &&
                Objects.equals(collateralDownloadDomains, contract.collateralDownloadDomains) &&
                Objects.equals(collateralInstallPrice, contract.collateralInstallPrice) &&
                Objects.equals(collateralInstallSoft, contract.collateralInstallSoft);
    }

    @Override
    public int hashCode() {
        return Objects.hash(externalId, type, contractType, rewardType, dt, endDt, isCancelled, isSigned, isFaxed,
                testMode, currency, vat, payTo, personId, firm, contract2Id, nds, collateralEndDt,
                collateralRewardType, collateralPayTo, collateralNds, collateralPartnerPct, collateralAgregatorPct,
                collateralMkbPrice, collateralSearchForms, collateralEndReason, collateralTailTime,
                collateralDistributionPlaces, collateralProductSearch, collateralProductSearchf,
                collateralProductOptions, collateralProductsDownload, collateralDownloadDomains,
                collateralInstallPrice, collateralInstallSoft);
    }

    public static class Builder {
        private String externalId;
        private String type;
        private Integer contractType;
        private Integer rewardType;
        private LocalDate dt;
        private LocalDate endDt;
        private LocalDate isCancelled;
        private LocalDate isSigned;
        private LocalDate isFaxed;
        private Integer testMode;
        private Integer currency;
        private String vat;
        private Integer payTo;
        private Long personId;
        private Integer firm;
        private Integer contract2Id;
        private Integer nds;

        public Builder withExternalId(String externalId) {
            this.externalId = externalId;
            return this;
        }

        public Builder withType(String type) {
            this.type = type;
            return this;
        }

        public Builder withContractType(Integer contractType) {
            this.contractType = contractType;
            return this;
        }

        public Builder withRewardType(Integer rewardType) {
            this.rewardType = rewardType;
            return this;
        }

        public Builder withDt(LocalDate dt) {
            this.dt = dt;
            return this;
        }

        public Builder withEndDt(LocalDate endDt) {
            this.endDt = endDt;
            return this;
        }

        public Builder withIsCancelled(LocalDate isCancelled) {
            this.isCancelled = isCancelled;
            return this;
        }

        public Builder withIsSigned(LocalDate isSigned) {
            this.isSigned = isSigned;
            return this;
        }

        public Builder withIsFaxed(LocalDate isFaxed) {
            this.isFaxed = isFaxed;
            return this;
        }

        public Builder withTestMode(Integer testMode) {
            this.testMode = testMode;
            return this;
        }

        public Builder withCurrency(Integer currency) {
            this.currency = currency;
            return this;
        }

        public Builder withVat(String vat) {
            this.vat = vat;
            return this;
        }

        public Builder withPayTo(Integer payTo) {
            this.payTo = payTo;
            return this;
        }

        public Builder withPersonId(Long personId) {
            this.personId = personId;
            return this;
        }

        public Builder withFirm(Integer firm) {
            this.firm = firm;
            return this;
        }

        public Builder withContract2Id(Integer contract2Id) {
            this.contract2Id = contract2Id;
            return this;
        }

        public Builder withNds(Integer nds) {
            this.nds = nds;
            return this;
        }

        public Contract build() {
            return new Contract(this);
        }
    }

    private static class OverrideField<T> {
        private boolean overridden;
        private T value;
        private T originalValue;

        OverrideField(T originalValue) {
            this.originalValue = originalValue;
            this.overridden = false;
            this.value = null;
        }

        T getValue() {
            return overridden
                    ? value
                    : originalValue;
        }

        void setValue(T value) {
            overridden = true;
            this.value = value;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof OverrideField)) {
                return false;
            }
            OverrideField<?> that = (OverrideField<?>) o;
            return overridden == that.overridden &&
                    Objects.equals(value, that.value);
        }

        @Override
        public int hashCode() {
            return Objects.hash(overridden, value);
        }
    }
}
