package ru.yandex.partner.libs.extservice.blackbox;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;

import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.impl.ArrayListF;
import ru.yandex.inside.passport.AbstractPassportUid;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.PassportUids;
import ru.yandex.inside.passport.YandexAccounts;
import ru.yandex.inside.passport.blackbox2.BlackboxRequestExecutor;
import ru.yandex.inside.passport.blackbox2.protocol.BlackboxException;
import ru.yandex.inside.passport.blackbox2.protocol.request.BlackboxRequest;
import ru.yandex.inside.passport.blackbox2.protocol.request.BlackboxRequestBuilder;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxAbstractResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxAvatar;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxBulkResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxDisplayName;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxResponseParseException;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxSessionIdException;
import ru.yandex.misc.ip.IpAddress;
import ru.yandex.partner.libs.common.HttpUtils;
import ru.yandex.partner.libs.common.RequestContextUtils;
import ru.yandex.passport.tvmauth.TvmClient;

@Service
public class BlackboxService {

    private static final Logger LOGGER = LoggerFactory.getLogger(BlackboxService.class);

    private static final int LANGUAGE_ATTRIBUTE = 34;
    private static final ListF<Integer> REQUIRED_ATTRIBUTES = new ArrayListF<>(List.of(LANGUAGE_ATTRIBUTE));

    public static final String USER_INFO_CACHE_OBJECT_KEY =
            BlackboxService.class.getCanonicalName().concat("userinfocache");
    public static final int CHUNK_SIZE = 500;

    private final BlackboxRequestExecutor blackboxRequestExecutor;
    private final TvmClient tvmClient;

    private final String blackboxTvmAlias;
    private final String defaultIpAddress;

    public BlackboxService(BlackboxRequestExecutor blackboxRequestExecutor,
                           TvmClient tvmClient,
                           @Value("${blackbox.tvmAlias:blackbox}") String blackboxTvmAlias,
                           @Value("${blackbox.clientIpStub:127.0.0.1}") String defaultIpAddress) {
        this.blackboxRequestExecutor = blackboxRequestExecutor;
        this.tvmClient = tvmClient;
        this.blackboxTvmAlias = blackboxTvmAlias;
        this.defaultIpAddress = defaultIpAddress;
    }

    public BlackboxUserInfo authenticateWithSessionId(String sessionId, String sslSessionId, String host)
            throws BlackboxSessionIdException {

        BlackboxRequestBuilder blackboxRequestBuilder =
                putCommonParametersToBuilder(BlackboxRequest.query().sessionId());

        blackboxRequestBuilder
                .host(host)
                .sessionId(sessionId)
                .sslSessionId(sslSessionId);

        BlackboxCorrectResponse response = blackboxRequestExecutor.execute(blackboxRequestBuilder.build()).getLeft()
                .getOrThrow();

        checkSessionIdResponseValid(response);

        BlackboxUserInfo blackboxUserInfo = convertResponseToUserInfo(response);
        cacheUserInfos(Map.of(blackboxUserInfo.getUid(), blackboxUserInfo));

        return blackboxUserInfo;
    }

    public BlackboxUserInfo getUserInfo(long uid) throws BlackboxException {
        return getUserInfos(List.of(uid)).get(uid);
    }

    public Map<Long, BlackboxUserInfo> getUserInfos(List<Long> uids) throws BlackboxException {
        Map<Long, BlackboxUserInfo> results = new HashMap<>(retrieveUserInfosFromCache(uids));
        List<Long> nonCached = uids.stream().filter(uid -> !results.containsKey(uid)).collect(Collectors.toList());
        if (!nonCached.isEmpty()) {
            results.putAll(getUserInfosNoCache(nonCached));
        }
        return results;
    }

    public Map<Long, BlackboxUserInfo> getUserInfosNoCache(List<Long> uids) throws BlackboxException {
        return Lists.partition(uids, CHUNK_SIZE).stream()
                .map(this::makeUserInfoRequest)
                .flatMap(map -> map.entrySet().stream())
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
    }

    private Map<Long, BlackboxUserInfo> makeUserInfoRequest(List<Long> uids) throws BlackboxException {
        Set<PassportUid> passportUids = uids.stream().distinct()
                .filter(PassportUids::isValid)
                .map(PassportUid::new).collect(Collectors.toSet());

        if (passportUids.isEmpty()) {
            return Collections.emptyMap();
        }

        BlackboxRequestBuilder blackboxRequestBuilder =
                putCommonParametersToBuilder(BlackboxRequest.query().userInfo());

        blackboxRequestBuilder
                .uid(new ArrayListF<>(passportUids));

        Either<BlackboxAbstractResponse, BlackboxBulkResponse> rawResponse =
                blackboxRequestExecutor.execute(blackboxRequestBuilder.build());

        Map<Long, BlackboxUserInfo> userInfos = convertBulkResponseToUserInfoMap(rawResponse);
        cacheUserInfos(userInfos);

        return uids.stream().collect(Collectors.toMap(
                uid -> uid, uid -> Optional.ofNullable(userInfos.get(uid)).orElseGet(BlackboxUserInfo::new)));
    }

    private BlackboxRequestBuilder putCommonParametersToBuilder(BlackboxRequestBuilder blackboxRequestBuilder) {
        return blackboxRequestBuilder
                .userIp(IpAddress.parse(getIPAddress()))
                .attributes(REQUIRED_ATTRIBUTES)
                .regName("yes")
                .addHeader(YandexAccounts.SERVICE_TICKET, getTvmServiceTicket());
    }

    private Map<Long, BlackboxUserInfo> retrieveUserInfosFromCache(Collection<Long> uids) {
        return RequestContextUtils.getObject(USER_INFO_CACHE_OBJECT_KEY, Map.class)
                .map(userInfoCache -> {
                    Map<Long, BlackboxUserInfo> userInfoMap = (Map<Long, BlackboxUserInfo>) userInfoCache;
                    LOGGER.info("Searching for UserInfos in cache");
                    return uids.stream()
                            .filter(userInfoMap::containsKey)
                            .map(userInfoMap::get)
                            .collect(Collectors.toMap(BlackboxUserInfo::getUid, userInfo -> userInfo));
                }).orElseGet(Collections::emptyMap);
    }

    private void cacheUserInfos(Map<Long, BlackboxUserInfo> userInfos) {
        if (RequestContextUtils.isRequestContextAvailable()) {
            Map<Long, BlackboxUserInfo> userInfoCache = RequestContextUtils.computeObject(
                    USER_INFO_CACHE_OBJECT_KEY, Map.class, HashMap::new);
            userInfoCache.putAll(userInfos);
        }
    }

    private void checkSessionIdResponseValid(BlackboxCorrectResponse response) throws BlackboxSessionIdException {
        BlackboxSessionIdException.BlackboxSessionIdStatus status =
                BlackboxSessionIdException.BlackboxSessionIdStatus.R.fromValue(response.getStatus());
        if (status == BlackboxSessionIdException.BlackboxSessionIdStatus.VALID) {
            if (response.getUid().toOptional().isEmpty() ||
                    response.getLogin().toOptional().isEmpty()) {
                // кривой ответ от blackbox - бросаем 500
                throw new BlackboxResponseParseException("Invalid response from BlackBox: " + response);
            }
            return;
        }
        // blackbox забраковал куку - бросаем 401
        throw new BlackboxSessionIdException(status, response.getError().getOrElse(""), null);
    }

    private BlackboxUserInfo convertResponseToUserInfo(BlackboxCorrectResponse response) {
        return new BlackboxUserInfo(
                getUidFromResponse(response).orElse(null), getLanguageFromResponse(response).orElse(null),
                getAvatarIdFromResponse(response).orElse(null)
        );
    }

    private Map<Long, BlackboxUserInfo> convertBulkResponseToUserInfoMap(
            Either<BlackboxAbstractResponse, BlackboxBulkResponse> rawResponse
    ) {
        Map<Long, BlackboxUserInfo> result;
        if (rawResponse.isLeft()) {
            BlackboxCorrectResponse response = rawResponse.getLeft().getOrThrow();
            BlackboxUserInfo blackboxUserInfo = convertResponseToUserInfo(response);
            result = (blackboxUserInfo.getUid() != null) ? Map.of(blackboxUserInfo.getUid(), blackboxUserInfo) :
                    Map.of();
        } else {
            MapF<PassportUid, BlackboxAbstractResponse> responses = rawResponse.getRight().get();
            result = responses.values().stream()
                    .map(blackboxAbstractResponse -> Optional.ofNullable(blackboxAbstractResponse)
                            .map(BlackboxAbstractResponse::getO).map(Option::getOrNull)
                            .map(this::convertResponseToUserInfo))
                    .filter(Optional::isPresent).map(Optional::get)
                    .filter(userInfo -> userInfo.getUid() != null)
                    .collect(Collectors.toMap(BlackboxUserInfo::getUid, userInfo -> userInfo));
        }
        return result;
    }

    private Optional<Long> getUidFromResponse(BlackboxCorrectResponse response) {
        return Optional.ofNullable(response)
                .map(BlackboxCorrectResponse::getUid).orElse(Option.empty())
                .map(AbstractPassportUid::getUid).toOptional();
    }

    private Optional<String> getLanguageFromResponse(BlackboxCorrectResponse response) {
        return Optional.ofNullable(response)
                .map(BlackboxCorrectResponse::getAttributes)
                .map(attributes -> attributes.getO(LANGUAGE_ATTRIBUTE)).map(Option::getOrNull);
    }

    private Optional<String> getAvatarIdFromResponse(BlackboxCorrectResponse response) {
        return Optional.ofNullable(response)
                .map(BlackboxCorrectResponse::getDisplayName).map(Option::getOrNull)
                .map(BlackboxDisplayName::getAvatar).map(Option::getOrNull)
                .map(BlackboxAvatar::getDefaultAvatarId);
    }

    private String getTvmServiceTicket() {
        return tvmClient.getServiceTicketFor(blackboxTvmAlias);
    }

    private String getIPAddress() {
        Optional<HttpServletRequest> request = RequestContextUtils.getRequest();
        if (request.isPresent()) {
            String ipAddress = HttpUtils.getIpAddrFromRequest(request.get());
            LOGGER.info("IP from request: {}", ipAddress);
            return ipAddress;
        }
        LOGGER.info("Default IP: {}", defaultIpAddress);
        return defaultIpAddress;
    }

}
