package ru.yandex.partner.libs.extservice.direct;

import java.net.URI;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.databind.PropertyNamingStrategies;
import com.fasterxml.jackson.databind.annotation.JsonNaming;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Service;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.util.UriComponentsBuilder;
import reactor.core.publisher.Mono;

import ru.yandex.partner.defaultconfiguration.rpc.DefaultRpcConfig;
import ru.yandex.partner.defaultconfiguration.rpc.RpcConfigProps;
import ru.yandex.partner.libs.common.PartnerRequestService;
import ru.yandex.partner.libs.tvm.TvmHeaders;
import ru.yandex.passport.tvmauth.TvmClient;

@Service
public class DirectIntapiServiceImpl implements DirectIntapiService {
    public static final String APP_INFO_PARSING_PATH = "/uac/app_info/parsing";

    private final WebClient webClient;
    private final TvmClient tvmClient;
    private final RpcConfigProps rpcConfigProps;
    private final DirectIntapiProperties directIntapiProperties;
    private final URI appInfoParsingUri;

    @Autowired
    public DirectIntapiServiceImpl(WebClient webClient,
                                   TvmClient tvmClient,
                                   DefaultRpcConfig rpcConfig,
                                   DirectIntapiProperties directIntapiProperties) {
        this.webClient = webClient;
        this.tvmClient = tvmClient;
        this.rpcConfigProps = rpcConfig;
        this.directIntapiProperties = directIntapiProperties;
        this.appInfoParsingUri = UriComponentsBuilder
                .fromUriString(directIntapiProperties.getUrl())
                .path(APP_INFO_PARSING_PATH)
                .build().toUri();
    }

    public AppInfo appInfoParsing(String appUrl) {
        String serviceTicket = tvmClient.getServiceTicketFor(directIntapiProperties.getTvmAlias());
        var responseEntity = webClient.post()
                .uri(appInfoParsingUri)
                .body(Mono.just(new Body(appUrl)), Body.class)
                .accept(MediaType.APPLICATION_JSON)
                .header(TvmHeaders.SERVICE_TICKET_HEADER_NAME, serviceTicket)
                .header("Host", appInfoParsingUri.getHost())
                .exchangeToMono(clientResponse -> clientResponse.toEntity(ResponseBody.class))
                .transform(PartnerRequestService.retryWithBackoff(rpcConfigProps))
                .block();

        if (responseEntity == null) {
            throw new RuntimeException("Uri: %s return NULL for %s".formatted(appInfoParsingUri.toString(), appUrl));
        }

        // если некорректный url передали он возвращает 400
        if (responseEntity.getStatusCode() == HttpStatus.BAD_REQUEST) {
            return null;
        }

        if (!responseEntity.getStatusCode().is2xxSuccessful()) {
            throw new RuntimeException("Uri: %s return HTTP_STATUS %d for %s".formatted(appInfoParsingUri.toString(),
                    responseEntity.getStatusCode().value(), appUrl));
        }

        var body = responseEntity.getBody();

        if (body == null) {
            return null;
        } else {
            return responseEntity.getBody().getAppInfo();
        }
    }

    private record Body(String url) {
    }

    @SuppressWarnings("unused")
    @JsonIgnoreProperties(ignoreUnknown = true)
    @JsonNaming(PropertyNamingStrategies.SnakeCaseStrategy.class)
    private static class ResponseBody {
        private AppInfo appInfo;

        public AppInfo getAppInfo() {
            return appInfo;
        }

        public void setAppInfo(AppInfo appInfo) {
            this.appInfo = appInfo;
        }
    }
}
