package ru.yandex.partner.libs.extservice.moneymap;

import java.net.URI;
import java.util.Objects;
import java.util.Optional;

import org.jooq.tools.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.codec.DecodingException;
import org.springframework.stereotype.Service;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.reactive.function.client.WebClientException;
import org.springframework.web.util.UriComponentsBuilder;

import ru.yandex.partner.libs.exceptions.I18nResponseStatusException;
import ru.yandex.partner.libs.i18n.MsgWithArgs;
import ru.yandex.partner.libs.tvm.TvmService;

import static ru.yandex.partner.libs.common.PartnerRequestService.retryWithBackoff;
import static ru.yandex.partner.libs.exceptions.HttpErrorStatusEnum.ERROR__INTERNAL;
import static ru.yandex.partner.libs.extservice.ThirdPartyServiceMsg.RESPONSE_IS_NOT_VALID_JSON;
import static ru.yandex.partner.libs.extservice.ThirdPartyServiceMsg.STATUS_CODE_NOT_OK;

@Service
public class MoneyMapService {
    private static final Logger LOGGER = LoggerFactory.getLogger(MoneyMapService.class);
    private static final String TVM_ALIAS_MONEYMAP = "moneymap";
    private static final String ABC_PATH = "/api/v1/abc/{abcId}";
    private static final String SUGGEST_PATH = "/api/v1/abc/suggest";
    private static final String TEXT_PARAM = "text";
    private static final String LIMIT_PARAM = "limit";
    private static final String OEBS_PARAM = "oebs";

    private final WebClient webClient;
    private final TvmService tvmService;
    private final MoneyMapRpcConfig rpcConfig;
    private final String moneyMapUrl;

    public MoneyMapService(
            WebClient webClient,
            TvmService tvmService,
            MoneyMapRpcConfig rpcConfig
    ) {
        this.webClient = webClient;
        this.tvmService = tvmService;
        this.rpcConfig = rpcConfig;
        this.moneyMapUrl = Objects.requireNonNull(rpcConfig.getUrl(), "MoneyMap url can't be null");
    }

    public JSONObject getServiceAbc(Integer abcId) {
        URI uri = UriComponentsBuilder.fromUriString(moneyMapUrl)
                .path(ABC_PATH)
                .build(abcId);

        LOGGER.info("Performing MoneyMap/ABC request {}", uri);
        return get(uri);
    }

    public JSONObject getSuggestAbc(String text, Integer limit, Boolean isOebs) {

        URI uri = UriComponentsBuilder.fromUriString(moneyMapUrl)
                .path(SUGGEST_PATH)
                .queryParam(TEXT_PARAM, text)
                .queryParam(LIMIT_PARAM, limit)
                .queryParam(OEBS_PARAM, isOebs)
                .build().toUri();

        LOGGER.info("Performing MoneyMap/ABC request {}", uri);
        return get(uri);
    }

    private JSONObject get(URI uri) {
        return webClient.get()
                .uri(uri)
                .headers(tvmService.attachTicketHeader(TVM_ALIAS_MONEYMAP))
                .retrieve()
                // при 4xx/5хх упадет с WebClientException по документации
                .toEntity(JSONObject.class)
                .onErrorMap(WebClientException.class, e -> {
                    LOGGER.error(e.toString());
                    return new I18nResponseStatusException(
                            ERROR__INTERNAL, MsgWithArgs.of(STATUS_CODE_NOT_OK, "moneymap"), e
                    );
                })
                .onErrorMap(DecodingException.class, e -> {
                    LOGGER.error("Failed to parse json for uri {}", uri);
                    return new I18nResponseStatusException(ERROR__INTERNAL, RESPONSE_IS_NOT_VALID_JSON, e);
                })
                .transform(retryWithBackoff(rpcConfig))
                .blockOptional()
                .flatMap(r -> Optional.ofNullable(r.getBody()))
                .orElseGet(() -> {
                    LOGGER.warn("Money map returned empty body with 200 response on {}", uri);
                    return new JSONObject();
                });
    }

}
