package ru.yandex.partner.libs.extservice.saas.service;

import java.nio.charset.StandardCharsets;
import java.util.Set;

import javax.annotation.Nullable;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.util.UriComponentsBuilder;
import reactor.core.publisher.Mono;

import ru.yandex.partner.defaultconfiguration.rpc.RpcConfigProps;
import ru.yandex.partner.libs.extservice.saas.response.SaasResponse;

import static ru.yandex.partner.libs.common.PartnerRequestService.retryWithBackoff;

@Service
public class UacSaasClientImpl implements UacSaasClient {
    private static final String SERVICE_PARAM = "service";
    private static final String TEXT_PARAM = "text";
    private static final String MS_PARAM = "ms";
    private static final String HR_PARAM = "hr";
    private static final String KPS_PARAM = "kps";
    private static final String LIMIT_PARAM = "numdoc";
    private static final String GTA_PARAM = "gta";
    private static final String SORT_PARAM = "how";
    private static final String SORT_FORMULA = "relev";
    private static final String TEMPLATE_PARAM = "template";

    private final WebClient webClient;
    private final String url;
    private final RpcConfigProps rpcConfigProps;

    @Autowired
    public UacSaasClientImpl(
            WebClient webClient,
            SaasUacRpcConfig saasUacRpcConfig
    ) {
        this.webClient = webClient;
        this.url = saasUacRpcConfig.getUrl();
        this.rpcConfigProps = saasUacRpcConfig;
    }

    /**
     * ?service=rmp_apps&kps=1&how=rlv&relev=60&numdoc=10&hr=da&text=andex
     * &ms=proto&template=z_title:"%25request%25*" | s_bundle_id:"%25request%25*"
     * Добавление параметра template включает режим опечаточника
     * Без режима опечаточника поиск будет по префиксу.
     * Группировка по стору ( google_play, itunes, appgallery)
     */
    @Override
    public Mono<ResponseEntity<SaasResponse>> performNonblockingRequest(String saasQuery, @Nullable String template,
                                                                        int limit, Set<String> gtaAttributes) {
        UriComponentsBuilder uriBuilder = UriComponentsBuilder
                .fromUriString(this.url)
                .path("/")
                .queryParam(SERVICE_PARAM, "rmp_apps")
                .queryParam(TEXT_PARAM, saasQuery)
                .queryParam(MS_PARAM, "proto")
                .queryParam(HR_PARAM, "json")
                .queryParam(KPS_PARAM, "1")
                .queryParam(SORT_PARAM, "rlv")
                .queryParam(SORT_FORMULA, String.valueOf(limit * 5))
                .queryParam(LIMIT_PARAM, limit);

        if (template != null) {
            uriBuilder.queryParam(TEMPLATE_PARAM, template);
        }

        gtaAttributes.forEach(attr -> uriBuilder.queryParam(GTA_PARAM, attr));

        return webClient.get()
                .uri(uriBuilder.build().toUri())
                .accept(MediaType.TEXT_PLAIN)
                .acceptCharset(StandardCharsets.UTF_8)
                .exchangeToMono(clientResponse -> clientResponse.toEntity(SaasResponse.class))
                .transform(retryWithBackoff(rpcConfigProps));
    }

}
