package ru.yandex.partner.libs.extservice.balance;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.time.LocalDate;
import java.util.List;
import java.util.stream.Collectors;

import okhttp3.HttpUrl;
import okhttp3.mockwebserver.MockResponse;
import okhttp3.mockwebserver.MockWebServer;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import ru.yandex.partner.libs.extservice.balance.method.partnercontract.BalancePartnerContract;
import ru.yandex.partner.libs.extservice.balance.method.partnercontract.Bank;
import ru.yandex.partner.libs.extservice.balance.method.partnercontract.Collateral;
import ru.yandex.partner.libs.extservice.balance.method.partnercontract.Contract;
import ru.yandex.partner.libs.extservice.balance.method.partnercontract.Person;


import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNull;

class BalanceServiceTest {
    private static final String HTTP_PATH = "/xmlrpc";

    private BalanceService balanceService;
    private MockWebServer server;

    @BeforeEach
    void setUp() {
        server = new MockWebServer();
        HttpUrl serverUrl = server.url(HTTP_PATH);
        var fakeConfig = new FakeBalanceRpcConfig();
        fakeConfig.setUrl(serverUrl.toString());
        balanceService = new SimpleBalanceService(fakeConfig);
    }

    @Test
    void getPartnerContracts() {
        String response = getResourceString("/balance/Balance.GetPartnerContracts.xml");
        server.enqueue(
                new MockResponse()
                        .setBody(response)
        );
        List<BalancePartnerContract> partnerContracts = balanceService.getPartnerContracts(1L);
        assertEquals(1, partnerContracts.size());
        assertEquals(new BalancePartnerContract(getPerson(), getContract(), getCollaterals()), partnerContracts.get(0));
    }

    @Test
    void getPartnerContractsWithNilXml() {
        String response = getResourceString("/balance/Balance.GetPartnerContracts.nil.xml");
        server.enqueue(
                new MockResponse()
                        .setBody(response)
        );
        List<BalancePartnerContract> partnerContracts = balanceService.getPartnerContracts(1L);
        assertEquals(1, partnerContracts.size());
        assertEquals(new BalancePartnerContract(getPerson(), getContract(), getCollaterals()), partnerContracts.get(0));
    }


    @Test
    void getBankByBik() {
        String response = getResourceString("/balance/Balance.GetBank.Bik.xml");
        server.enqueue(
                new MockResponse()
                        .setBody(response)
        );

        assertEquals(
                new Bank(true, "044525225", "bik", "ПАО СБЕРБАНК"),
                balanceService.getBankByBik("044525225")
        );
    }

    @Test
    void getBankBySwift() {
        String response = getResourceString("/balance/Balance.GetBank.Swift.xml");
        server.enqueue(
                new MockResponse()
                        .setBody(response)
        );

        assertEquals(
                new Bank(true, "BPKOPLPWXXX", "swift", "PKO BANK POLSKI S.A."),
                balanceService.getBankBySwift("BPKOPLPWXXX")
        );
    }

    @Test
    void getBankByBik404() {
        server.enqueue(
                new MockResponse()
                        .setResponseCode(404)
        );

        assertNull(balanceService.getBankByBik("bik"));
    }

    @Test
    void getBankBySwift404() {
        server.enqueue(
                new MockResponse()
                        .setResponseCode(404)
        );

        assertNull(balanceService.getBankBySwift("swift"));
    }

    @Test
    void getBankByBik400() {
        server.enqueue(
                new MockResponse()
                        .setResponseCode(400)
        );

        assertNull(balanceService.getBankByBik("bik"));
    }

    @Test
    void getBankBySwift400() {
        server.enqueue(
                new MockResponse()
                        .setResponseCode(400)
        );

        assertNull(balanceService.getBankBySwift("swift"));
    }

    @Test
    void getBankByBik500() {
        server.enqueue(
                new MockResponse()
                        .setResponseCode(500)
        );

        assertNull(balanceService.getBankByBik("bik"));
    }

    @Test
    void getBankBySwift500() {
        server.enqueue(
                new MockResponse()
                        .setResponseCode(500)
        );

        assertNull(balanceService.getBankBySwift("swift"));
    }

    @AfterEach
    void tearDown() throws IOException {
        server.shutdown();
    }


    private Person getPerson() {
        return new Person.Builder()
                .withId(6828875L)
                .withClientId(1560494L)
                .withName("РА ИНДЕКС 20")
                .withLongname("ООО \"РА \"Индекс 20\"")
                .withPhone("+7 495 785-17-00")
                .withEmail("docsyandex@rambler-co.ru;rambler-p@yandex.ru")
                .withFax("fax")
                .withLegaladdress("legaladdress")
                .withRepresentative("representative")
                .withAccount("40702810338000165465")
                .withAuthorityDocType("Устав")
                .withBenBank("ben_bank")
                .withBenAccount("")
                .withBik("044525225")
                .withIban("iban")
                .withInn("7725847502")
                .withKpp("772501001")
                .withOther("other")
                .withPersonAccount("person_account")
                .withPostaddress("postaddress")
                .withSignerPersonName("Тадевосян М. К.")
                .withSignerPositionName("Генеральный директор")
                .withSwift("swift")
                .withYamoneyWallet("yamoney_wallet")
                .withPayoneerWallet("")
                .withType("ur")
                .build();
    }

    private Contract getContract() {
        return new Contract.Builder()
                .withExternalId("ИНДЕКС_Д_2018/2603")
                .withType("PARTNERS")
                .withContractType(6)
                .withRewardType(2)
                .withDt(LocalDate.of(2019, 1, 1))
                .withEndDt(LocalDate.of(2020, 8, 6))
                .withIsCancelled(LocalDate.of(2099, 1, 1))
                .withIsSigned(LocalDate.of(2018, 12, 28))
                .withIsFaxed(LocalDate.of(2019, 6, 7))
                .withTestMode(1)
                .withCurrency(643)
                .withVat("vat")
                .withPayTo(1)
                .withPersonId(6828875L)
                .withFirm(1)
                .withNds(18)
                .withContract2Id(555011)
                .build();
    }

    private List<Collateral> getCollaterals() {
        return List.of(
                new Collateral.Builder()
                        .withCollateralTypeId(2040)
                        .withBalanceClassType("COLLATERAL")
                        .withDt(LocalDate.of(2019, 1, 25))
//                .withEndDt()
                        .withIsCancelled(LocalDate.of(2099, 1, 1))
                        .withIsSigned(LocalDate.of(2019, 3, 19))
                        .withIsFaxed(LocalDate.of(2019, 8, 9))
//                .withNds()
//                .withPartnerPct()
//                .withAgregatorPct()
//                .withMkbPrice()
//                .withSearchForms()
//                .withEndReason()
//                .withPayTo()
//                .withTailTime()
//                .withDistributionPlaces()
//                .withProductSearch()
//                .withProductSearchf()
//                .withProductOptions()
//                .withProductsDownload()
//                .withDownloadDomains()
//                .withInstallPrice()
//                .withInstallSoft()
//                .withRewardType()
                        .build()
        );
    }

    private String getResourceString(String resourcePath) {
        InputStream inputStream = this.getClass().getResourceAsStream(resourcePath);

        return (new BufferedReader(new InputStreamReader(inputStream))).lines().collect(Collectors.joining());
    }
}
