package ru.yandex.partner.libs.extservice.blackbox;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.Option;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.BlackboxRequestExecutor;
import ru.yandex.inside.passport.blackbox2.protocol.request.BlackboxRequest;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxAvatar;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxDisplayName;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxSessionIdException;
import ru.yandex.misc.algo.ht.examples.OpenHashMap;
import ru.yandex.passport.tvmauth.TvmClient;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.doCallRealMethod;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

public class BlackboxServiceTest {

    public static final String FAKE_SERVICE_TICKET = "fake-service-ticket";

    public static final String SESSION_ID_VALUE = "session_1_value";
    public static final String SESSIONID2_VALUE = "session_2_value";

    private static final String DEFAULT_IP = "1.1.1.1";
    private static final String SERVER_HOST = "my.host";
    public static final String LANGUAGE = "ru";
    public static final String TVM_ALIAS = "blackbox";

    private BlackboxRequestExecutor blackboxRequestExecutor;
    private TvmClient tvmClient;

    private BlackboxService blackboxService;
    public static final int USER_ID = 274;
    public static final String USER_LOGIN = "usser";
    public static final String AVATAR_ID = "avatar-id";

    @BeforeEach
    void init() {
        blackboxRequestExecutor = mock(BlackboxRequestExecutor.class);

        tvmClient = mock(TvmClient.class);
        doReturn(FAKE_SERVICE_TICKET).when(tvmClient).getServiceTicketFor(TVM_ALIAS);

        blackboxService = new BlackboxService(blackboxRequestExecutor, tvmClient, TVM_ALIAS, DEFAULT_IP);
    }

    @Test
    public void testAuthenticateWithSessionIdSuccess() {
        mockBlackboxSessionIdResponse(BlackboxSessionIdException.BlackboxSessionIdStatus.VALID);

        BlackboxUserInfo blackboxUserInfo = blackboxService.authenticateWithSessionId(
                SESSION_ID_VALUE, SESSIONID2_VALUE, SERVER_HOST);

        assertEquals(USER_ID, blackboxUserInfo.getUid());
        assertEquals(LANGUAGE, blackboxUserInfo.getLanguage());
    }

    @Test
    public void testAuthenticateWithSessionIdFail() {
        mockBlackboxSessionIdResponse(BlackboxSessionIdException.BlackboxSessionIdStatus.INVALID);

        BlackboxSessionIdException exception = assertThrows(BlackboxSessionIdException.class, () ->
                blackboxService.authenticateWithSessionId(SESSION_ID_VALUE, SESSIONID2_VALUE, SERVER_HOST));

        assertEquals(BlackboxSessionIdException.BlackboxSessionIdStatus.INVALID, exception.getStatus());
    }

    @Test
    public void testAuthenticateWithSessionIdFailTrow() {
        prepareBlackboxSessionIdThrow(BlackboxSessionIdException.BlackboxSessionIdStatus.EXPIRED);

        BlackboxSessionIdException exception = assertThrows(BlackboxSessionIdException.class, () ->
                blackboxService.authenticateWithSessionId(SESSION_ID_VALUE, SESSIONID2_VALUE, SERVER_HOST));

        assertEquals(BlackboxSessionIdException.BlackboxSessionIdStatus.EXPIRED, exception.getStatus());
    }

    @Test
    public void testGetLanguageSuccess() {
        mockBlackboxSessionIdResponse(BlackboxSessionIdException.BlackboxSessionIdStatus.VALID);

        String language = blackboxService.getUserInfo(USER_ID).getLanguage();

        assertEquals(LANGUAGE, language);
    }

    private void mockBlackboxSessionIdResponse(BlackboxSessionIdException.BlackboxSessionIdStatus status) {
        BlackboxCorrectResponse blackboxCorrectResponse = createMockedResponse(status);

        doReturn(Either.left(blackboxCorrectResponse))
                .when(blackboxRequestExecutor).execute(any(BlackboxRequest.class));
    }

    private void prepareBlackboxSessionIdThrow(BlackboxSessionIdException.BlackboxSessionIdStatus status) {
        BlackboxSessionIdException blackboxSessionIdException =
                new BlackboxSessionIdException(status, null, "");

        doThrow(blackboxSessionIdException).when(blackboxRequestExecutor).execute(any(BlackboxRequest.class));
    }

    private BlackboxCorrectResponse createMockedResponse(BlackboxSessionIdException.BlackboxSessionIdStatus status) {
        BlackboxCorrectResponse blackboxCorrectResponse = mock(BlackboxCorrectResponse.class);
        doReturn(blackboxCorrectResponse).when(blackboxCorrectResponse).getOrThrow();
        doCallRealMethod().when(blackboxCorrectResponse).getO();
        when(blackboxCorrectResponse.getStatus())
                .thenReturn(status.value());
        when(blackboxCorrectResponse.getUid()).thenReturn(Option.ofNullable(new PassportUid(USER_ID)));
        when(blackboxCorrectResponse.getLogin()).thenReturn(Option.ofNullable(BlackboxServiceTest.USER_LOGIN));

        OpenHashMap<Integer, String> attributes = new OpenHashMap<>();
        attributes.put(34, LANGUAGE);
        when(blackboxCorrectResponse.getAttributes()).thenReturn(attributes);

        BlackboxAvatar blackboxAvatar = mock(BlackboxAvatar.class);
        when(blackboxAvatar.getDefaultAvatarId()).thenReturn(AVATAR_ID);
        BlackboxDisplayName blackboxDisplayName = new BlackboxDisplayName(null, Option.empty(),
                Option.ofNullable(blackboxAvatar), Option.empty(), Option.empty());
        when(blackboxCorrectResponse.getDisplayName()).thenReturn(Option.ofNullable(blackboxDisplayName));

        when(blackboxCorrectResponse.getError()).thenReturn(Option.empty());
        return blackboxCorrectResponse;
    }

}
