package ru.yandex.partner.libs.extservice.direct;

import java.io.IOException;
import java.time.Duration;

import okhttp3.HttpUrl;
import okhttp3.mockwebserver.MockResponse;
import okhttp3.mockwebserver.MockWebServer;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.web.reactive.function.client.WebClient;

import ru.yandex.partner.defaultconfiguration.rpc.DefaultRpcConfig;
import ru.yandex.passport.tvmauth.TvmClient;

import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.mock;

class DirectIntapiServiceImplTest {
    private static MockWebServer mockWebServer;
    private static DirectIntapiProperties directIntapiProperties;
    private static DefaultRpcConfig defaultRpcConfig;
    private DirectIntapiServiceImpl directIntapiService;

    @BeforeAll
    static void beforeAll() {
        mockWebServer = new MockWebServer();
        HttpUrl serverUrl = mockWebServer.url(DirectIntapiServiceImpl.APP_INFO_PARSING_PATH);

        directIntapiProperties = new DirectIntapiProperties();
        directIntapiProperties.setUrl(serverUrl.toString());
        directIntapiProperties.setTvmAlias("mock_direct_intapi");

        defaultRpcConfig = new DefaultRpcConfig();
        defaultRpcConfig.setTimeout(Duration.ofSeconds(3));
        defaultRpcConfig.setRetries(2);
        defaultRpcConfig.setDelay(Duration.ofMillis(100));
    }

    @AfterAll
    static void afterAll() throws IOException {
        mockWebServer.shutdown();
    }

    @BeforeEach
    void setUp() {
        var tvmClient = mock(TvmClient.class);
        doReturn("generated_service_ticket").when(tvmClient).getServiceTicketFor("mock_direct_intapi");
        directIntapiService = new DirectIntapiServiceImpl(
                WebClient.create(),
                tvmClient,
                defaultRpcConfig,
                directIntapiProperties
        );
    }

    @Test
    void appInfoParsing2xx() {
        String body = """
                {
                    "app_info": {
                        "id": "8541870732769584190",
                        "app_id": "com.app.test",
                        "bundle_id": "com.app.test",
                        "title": "App Test",
                        "icon_url": "https://test.app.com/",
                        "language": "ru",
                        "platform": 1,
                        "region": "ru",
                        "vendor": "Somebody",
                        "rating": 4.4,
                        "reviews": 24,
                        "age_limit": 3,
                        "min_os_version": null,
                        "url": "https://play.google.com/store/apps/details?hl=ru&gl=ru&id=com.app.test",
                        "currency": "RUB",
                        "price": 0.0
                    }
                }
                """;

        mockWebServer.enqueue(
                new MockResponse()
                        .setBody(body)
                        .addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE)
        );

        var expectedAppInfo = new AppInfo();
        expectedAppInfo.setTitle("App Test");
        expectedAppInfo.setIconUrl("https://test.app.com/");
        expectedAppInfo.setPlatform(1);
        expectedAppInfo.setBundleId("com.app.test");
        expectedAppInfo.setVendor("Somebody");
        expectedAppInfo.setRegion("ru");
        expectedAppInfo.setUrl("https://play.google.com/store/apps/details?hl=ru&gl=ru&id=com.app.test");
        expectedAppInfo.setAppId("com.app.test");

        var appInfo = directIntapiService.appInfoParsing("app_url");

        Assertions.assertEquals(expectedAppInfo, appInfo);
    }

    @Test
    void appInfoParsing400() {
        mockWebServer.enqueue(
                new MockResponse().setResponseCode(400)
        );

        var appInfo =  directIntapiService.appInfoParsing("app_url");

        Assertions.assertNull(appInfo);
    }

    @Test
    void appInfoParsing4xx() {
        mockWebServer.enqueue(
                new MockResponse().setResponseCode(404)
        );

        Assertions.assertThrows(RuntimeException.class, () -> directIntapiService.appInfoParsing("app_url"));
    }

    @Test
    void appInfoParsing5xx() {
        mockWebServer.enqueue(
                new MockResponse().setResponseCode(500)
        );

        Assertions.assertThrows(RuntimeException.class, () -> directIntapiService.appInfoParsing("app_url"));
    }
}
