package ru.yandex.partner.libs.extservice.moneymap;

import java.io.IOException;
import java.time.Duration;
import java.util.Map;
import java.util.stream.IntStream;

import com.fasterxml.jackson.databind.ObjectMapper;
import okhttp3.HttpUrl;
import okhttp3.mockwebserver.MockResponse;
import okhttp3.mockwebserver.MockWebServer;
import org.jooq.tools.json.JSONObject;
import org.json.JSONException;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;
import org.skyscreamer.jsonassert.JSONAssert;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.web.reactive.function.client.WebClient;

import ru.yandex.partner.libs.exceptions.I18nResponseStatusException;
import ru.yandex.partner.libs.tvm.TvmHeaders;
import ru.yandex.partner.libs.tvm.TvmService;

import static org.mockito.Mockito.mock;

public class MoneyMapServiceTest {
    static final String MOCK_URL = "/moneymap";

    static MockWebServer mockWebServer;
    static MoneyMapService moneyMapService;
    static ObjectMapper mapper = new ObjectMapper();

    @BeforeAll
    static void setUp() {
        mockWebServer = new MockWebServer();
        HttpUrl serverUrl = mockWebServer.url(MOCK_URL);

        FakeMoneyMapRpcConfig fakeConfig = new FakeMoneyMapRpcConfig();
        fakeConfig.setUrl(serverUrl.toString());
        fakeConfig.setTimeout(Duration.ofSeconds(3));
        fakeConfig.setRetries(2);
        fakeConfig.setDelay(Duration.ofMillis(100));

        TvmService mockTvmService = mock(TvmService.class);
        Mockito.when(mockTvmService.attachTicketHeader("moneymap"))
                .thenReturn(h -> h.set(TvmHeaders.SERVICE_TICKET_HEADER_NAME, "mocked-service-ticket"));
        moneyMapService = new MoneyMapService(
                WebClient.create(),
                mockTvmService,
                fakeConfig
        );
    }

    @AfterAll
    static void tearDown() throws IOException {
        mockWebServer.shutdown();
    }

    @Test
    void getSuggestTest() throws IOException, InterruptedException, JSONException {
        String body = "{\"count\":10,\"services\":[{}]}";
        mockWebServer.enqueue(
                new MockResponse().setBody(body)
                        .addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE)
        );
        String expected = "{\"data\":{\"count\":10,\"services\":[{}]}}";
        Map<String, JSONObject> answer = Map.of(
                "data",
                moneyMapService.getSuggestAbc("partner", 1, true)
        );
        String actual = mapper.writeValueAsString(answer);
        JSONAssert.assertEquals(expected, actual, true);

        var req = mockWebServer.takeRequest();
        Assertions.assertTrue(req.getPath().contains("api/v1/abc/suggest?text=partner&limit=1&oebs=true"));
    }


    @Test
    void getServiceTest() throws IOException, InterruptedException, JSONException {
        String body = "{\"service\":{\"slug\":\"fake_abc\",\"synced_with_oebs\":true,\"eng_name\":\"No ABC\"," +
                "\"ru_name\":\"НЕТ разметки в ABC\",\"abc_id\":-1}}";
        mockWebServer.enqueue(
                new MockResponse().setBody(body)
                        .addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE)
        );
        String expected = "{\"data\":{\"service\":{\"slug\":\"fake_abc\",\"synced_with_oebs\":true,\"eng_name\":\"No " +
                "ABC\"," + "\"ru_name\":\"НЕТ разметки в ABC\",\"abc_id\":-1}}}";
        Map<String, JSONObject> answer = Map.of(
                "data",
                moneyMapService.getServiceAbc(-1)
        );
        String actual = mapper.writeValueAsString(answer);
        JSONAssert.assertEquals(expected, actual, true);

        var req = mockWebServer.takeRequest();
        Assertions.assertTrue(req.getPath().contains("api/v1/abc/-1"));
    }

    @Test
    void handle5xxTest() throws InterruptedException {
        // мы должны ответить на запрос и каждый ретрай
        enqueueTimes(
                new MockResponse()
                        .setBody("Service unavailable")
                        .setResponseCode(500)
                        .addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE),
                3
        );
        Assertions.assertThrows(I18nResponseStatusException.class, () -> {
            var result = moneyMapService.getSuggestAbc("exampl", 10, false);
        });

        takeRequests(3);
    }

    @Test
    void invalidJsonTest() throws InterruptedException {
        enqueueTimes(
                new MockResponse().setBody("I'm invalid json")
                        .addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE),
                3
        );
        Assertions.assertThrows(I18nResponseStatusException.class, () -> {
            var result = moneyMapService.getServiceAbc(123);
        });

        takeRequests(3);
    }

    private void enqueueTimes(MockResponse response, Integer times) {
        IntStream.range(0, times).forEach(i -> mockWebServer.enqueue(response));
    }

    private void takeRequests(Integer times) throws InterruptedException {
        for (int i = 0; i < times; i++) {
            mockWebServer.takeRequest();
        }
    }
}
