package ru.yandex.direct.hourglass.mysql.schedulerinstances;

import java.time.Duration;
import java.time.ZoneId;
import java.util.List;
import java.util.Objects;

import javax.annotation.Nullable;

import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.DatePart;
import org.jooq.impl.DSL;

import ru.yandex.direct.hourglass.InstanceId;
import ru.yandex.direct.hourglass.InstanceMeta;
import ru.yandex.direct.hourglass.client.SchedulerInstance;
import ru.yandex.direct.hourglass.implementations.InstanceIdImpl;
import ru.yandex.direct.hourglass.implementations.updateschedule.SchedulerInstanceImpl;
import ru.yandex.direct.hourglass.updateschedule.SchedulerInstancesRepository;

import static org.jooq.impl.DSL.count;
import static org.jooq.impl.DSL.currentLocalDateTime;
import static org.jooq.impl.DSL.localDateTimeSub;
import static ru.yandex.partner.dbschema.partner.Tables.SCHEDULER_INSTANCES;

public class MysqlSchedulerInstancesRepository implements SchedulerInstancesRepository {

    private static final long INSTANCE_IS_MAIN = 1L;
    private static final long INSTANCE_IS_NOT_MAIN = 0L;
    private static final Duration DEFAULT_INSTANCES_EXPIRATION = Duration.ofDays(1);
    private static final Duration DEFAULT_HEARTBEAT_EXPIRATION = Duration.ofMinutes(5);
    private static final int DEFAULT_LEADER_VOTING_LOWER_BOUND = 1;
    private final DSLContext dslContext;
    private final Duration heartbeatExpiration;
    private final Duration instancesExpiration;
    private final int leaderVotingLowerBound;

    /**
     * @param heartbeatExpiration    инстансы, у которых heartbeatTime не обновлялся больше указанного времени, не
     *                               участвуют в выборе версии
     * @param instancesExpiration    если поле heartbeatTime у какого-либо инстанса не обновлялся больше указанного
     *                               времени, его
     *                               можно удалить из таблицы
     * @param leaderVotingLowerBound минимальное количество голосов, чтобы версия была назначена лидером
     */
    private MysqlSchedulerInstancesRepository(DSLContext dslContext, Duration heartbeatExpiration,
                                              Duration instancesExpiration, int leaderVotingLowerBound) {
        this.dslContext = dslContext;
        this.heartbeatExpiration = heartbeatExpiration;
        this.instancesExpiration = instancesExpiration;
        this.leaderVotingLowerBound = leaderVotingLowerBound;
    }


    @Override
    public void markInstanceAsMain(InstanceId instanceId) {
        setMainField(instanceId, INSTANCE_IS_MAIN);
    }

    @Override
    public void markInstanceAsNotMain(InstanceId instanceId) {
        setMainField(instanceId, INSTANCE_IS_NOT_MAIN);
    }

    @Override
    public boolean isLeaderVersion(String version) {
        removeOldInstances();

        var countField = count().as("count");

        var versionToCnt = dslContext.select(SCHEDULER_INSTANCES.VERSION, countField)
                .from(SCHEDULER_INSTANCES)
                .where(SCHEDULER_INSTANCES.IS_MAIN.eq(INSTANCE_IS_MAIN)
                        .and(isInstanceAliveCondition()))
                .groupBy(SCHEDULER_INSTANCES.VERSION)
                .orderBy(countField.desc())
                .fetch();

        if (versionToCnt.isEmpty() || (versionToCnt.size() > 1 && versionToCnt.get(0).getValue(countField)
                .equals(versionToCnt.get(1).getValue(countField)))) {
            return false;
        }

        return versionToCnt.get(0).get(SCHEDULER_INSTANCES.VERSION).equals(version)
                && versionToCnt.get(0).get(countField) >= leaderVotingLowerBound;
    }


    @Override
    public void addInstance(InstanceId instanceId, String version, @Nullable InstanceMeta instanceMeta) {
        var instanceMetaString = Objects.nonNull(instanceMeta) ? instanceMeta.getString() : "";
        addInstance(instanceId, version, instanceMetaString);
    }

    private void addInstance(InstanceId instanceId, String version, String instanceMetaString) {
        dslContext.insertInto(SCHEDULER_INSTANCES).columns(SCHEDULER_INSTANCES.INSTANCE_ID,
                SCHEDULER_INSTANCES.VERSION, SCHEDULER_INSTANCES.IS_MAIN, SCHEDULER_INSTANCES.META)
                .values(instanceId.toString(), version, 1L, instanceMetaString)
                .execute();
    }


    @Override
    public void removeInstance(InstanceId instanceId) {
        dslContext.deleteFrom(SCHEDULER_INSTANCES).where(SCHEDULER_INSTANCES.INSTANCE_ID.eq(instanceId.toString()))
                .execute();
    }

    @Override
    public void pingInstance(InstanceId instanceId) {
        dslContext.update(SCHEDULER_INSTANCES)
                .set(SCHEDULER_INSTANCES.HEARTBEAT_TIME, currentLocalDateTime())
                .where(SCHEDULER_INSTANCES.INSTANCE_ID.eq(instanceId.toString()))
                .execute();
    }

    private Condition isInstanceAliveCondition() {
        return SCHEDULER_INSTANCES.HEARTBEAT_TIME.ge(
                localDateTimeSub(currentLocalDateTime(), heartbeatExpiration.getSeconds(), DatePart.SECOND));
    }

    @Override
    public List<SchedulerInstance> getSchedulerInstancesInfo() {
        var isActiveFieldName = "isActive";

        var isActive =
                DSL.when(isInstanceAliveCondition(), Boolean.TRUE).otherwise(Boolean.FALSE).as(isActiveFieldName);

        return dslContext.select(
                isActive,
                SCHEDULER_INSTANCES.INSTANCE_ID,
                SCHEDULER_INSTANCES.VERSION,
                SCHEDULER_INSTANCES.IS_MAIN,
                SCHEDULER_INSTANCES.HEARTBEAT_TIME,
                SCHEDULER_INSTANCES.META)
                .from(SCHEDULER_INSTANCES)
                .fetch(r -> new SchedulerInstanceImpl(
                        new InstanceIdImpl(r.get(SCHEDULER_INSTANCES.INSTANCE_ID)),
                        r.get(SCHEDULER_INSTANCES.VERSION),
                        r.get(SCHEDULER_INSTANCES.IS_MAIN) == 1L,
                        r.get(SCHEDULER_INSTANCES.HEARTBEAT_TIME).atZone(ZoneId.systemDefault()).toInstant(),
                        r.get(SCHEDULER_INSTANCES.META),
                        (boolean) r.get(isActiveFieldName)));
    }

    private void removeOldInstances() {
        dslContext.deleteFrom(SCHEDULER_INSTANCES)
                .where(SCHEDULER_INSTANCES.HEARTBEAT_TIME.le((localDateTimeSub(currentLocalDateTime(),
                        instancesExpiration.getSeconds(), DatePart.SECOND))))
                .execute();
    }

    private void setMainField(InstanceId instanceId, long isMain) {
        dslContext.update(SCHEDULER_INSTANCES)
                .set(SCHEDULER_INSTANCES.IS_MAIN, isMain)
                .where(SCHEDULER_INSTANCES.INSTANCE_ID.eq(instanceId.toString()))
                .execute();
    }

    public static class Builder {
        private final DSLContext dslContext;
        private Duration heartbeatExpiration = DEFAULT_HEARTBEAT_EXPIRATION;
        private Duration instancesExpiration = DEFAULT_INSTANCES_EXPIRATION;
        private int leaderVotingLowerBound = DEFAULT_LEADER_VOTING_LOWER_BOUND;

        public Builder(DSLContext dslContext) {
            this.dslContext = dslContext;
        }

        public Builder withHeartbeatExpiration(Duration heartbeatExpiration) {
            this.heartbeatExpiration = heartbeatExpiration;
            return this;
        }

        public Builder withInstancesExpiration(Duration instancesExpiration) {
            this.instancesExpiration = instancesExpiration;
            return this;
        }

        public Builder withLeaderVotingLowerBound(int leaderVotingLowerBound) {
            this.leaderVotingLowerBound = leaderVotingLowerBound;
            return this;
        }

        public MysqlSchedulerInstancesRepository build() {
            return new MysqlSchedulerInstancesRepository(dslContext, heartbeatExpiration, instancesExpiration,
                    leaderVotingLowerBound);
        }
    }
}
