package ru.yandex.direct.hourglass.mysql.storage;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.DatePart;
import org.jooq.Field;
import org.jooq.InsertSetMoreStep;
import org.jooq.InsertSetStep;
import org.jooq.Record;
import org.jooq.RecordMapper;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;

import ru.yandex.direct.hourglass.HourglassProperties;
import ru.yandex.direct.hourglass.InstanceId;
import ru.yandex.direct.hourglass.implementations.InstanceIdImpl;
import ru.yandex.direct.hourglass.implementations.TaskProcessingResultImpl;
import ru.yandex.direct.hourglass.implementations.updateschedule.ScheduleRecord;
import ru.yandex.direct.hourglass.storage.Job;
import ru.yandex.direct.hourglass.storage.JobStatus;
import ru.yandex.direct.hourglass.storage.PrimaryId;
import ru.yandex.direct.hourglass.storage.Storage;
import ru.yandex.direct.hourglass.storage.TaskId;
import ru.yandex.direct.hourglass.storage.Update;
import ru.yandex.direct.hourglass.storage.implementations.Find;
import ru.yandex.direct.hourglass.storage.implementations.JobImpl;
import ru.yandex.direct.hourglass.storage.implementations.TaskIdImpl;
import ru.yandex.partner.dbschema.partner.tables.records.ScheduledTasksRecord;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.hourglass.storage.JobStatus.ARCHIVED;
import static ru.yandex.direct.hourglass.storage.JobStatus.LOCKED;
import static ru.yandex.direct.hourglass.storage.JobStatus.READY;
import static ru.yandex.direct.hourglass.storage.JobStatus.STOPPED;
import static ru.yandex.partner.dbschema.partner.enums.ScheduledTasksStatus.Deleted;
import static ru.yandex.partner.dbschema.partner.enums.ScheduledTasksStatus.New;
import static ru.yandex.partner.dbschema.partner.enums.ScheduledTasksStatus.Paused;
import static ru.yandex.partner.dbschema.partner.enums.ScheduledTasksStatus.Running;
import static ru.yandex.partner.dbschema.partner.tables.ScheduledTasks.SCHEDULED_TASKS;

public class StorageImpl implements Storage {
    private static final String UNIVERSAL_VERSION = "";
    private final DSLContext ctxt;
    private final InstanceId schedulerId;
    private final JobRecordMapper jobRecordMapper;
    private final HourglassProperties configuration;
    private final String version;

    public StorageImpl(DSLContext ctxt, InstanceId schedulerId, HourglassProperties configuration, String version) {
        this.ctxt = ctxt;
        this.schedulerId = schedulerId;
        this.configuration = configuration;
        this.jobRecordMapper = new JobRecordMapper();
        this.version = version;
    }

    public StorageImpl(DSLContext ctxt, InstanceId schedulerId, HourglassProperties configuration) {
        this(ctxt, schedulerId, configuration, UNIVERSAL_VERSION);
    }


    private void setFieldsFromStatus(
            ScheduledTasksRecord scheduledTasksRecord, JobStatus status) {
        switch (status) {
            case ARCHIVED:
                scheduledTasksRecord.setStatus(Deleted);
                scheduledTasksRecord.setInstanceId(null);
                scheduledTasksRecord.setHeartbeatTime(null);
                break;

            case READY:
                scheduledTasksRecord.setStatus(New);
                scheduledTasksRecord.setInstanceId(null);
                scheduledTasksRecord.setHeartbeatTime(null);
                break;

            case LOCKED:
                scheduledTasksRecord.setStatus(Running);
                scheduledTasksRecord.setHeartbeatTime(LocalDateTime.now());
                scheduledTasksRecord.setInstanceId(schedulerId.toString());
                break;

            case STOPPED:
                scheduledTasksRecord.setStatus(Paused);
                scheduledTasksRecord.setInstanceId(null);
                scheduledTasksRecord.setHeartbeatTime(null);
                break;

            case EXPIRED:
                throw new IllegalArgumentException("Moving to EXPIRED state by hands is strictly prohibited by UNA");

            default:
                throw new IllegalArgumentException("Not implemented");
        }
    }

    private Condition statusToCondition(JobStatus jobStatus) {
        Condition condition;

        switch (jobStatus) {
            case ARCHIVED:
                condition = SCHEDULED_TASKS.STATUS.eq(Deleted);
                break;

            case READY:
                var startCondition = SCHEDULED_TASKS.STATUS.eq(New)
                        .and(SCHEDULED_TASKS.INSTANCE_ID.isNull());
                condition = addVersionCondition(startCondition);
                break;

            case EXPIRED:
                condition = DSL.and(
                        SCHEDULED_TASKS.INSTANCE_ID.isNotNull(),
                        SCHEDULED_TASKS.STATUS.eq(Running),
                        SCHEDULED_TASKS.HEARTBEAT_TIME.le(DSL.localDateTimeSub(DSL.currentLocalDateTime(),
                                configuration.getTaskHeartbeatExpiration().toSeconds(),
                                DatePart.SECOND)));
                break;

            case STOPPED:
                condition = SCHEDULED_TASKS.STATUS.eq(Paused);
                break;

            case LOCKED:
                condition = DSL.and(
                        SCHEDULED_TASKS.STATUS.eq(Running),
                        SCHEDULED_TASKS.INSTANCE_ID.eq(schedulerId.toString()));
                break;

            default:
                throw new IllegalStateException("Unknown  state " + jobStatus);
        }

        return condition;
    }

    private Condition addVersionCondition(Condition condition) {
        if (!UNIVERSAL_VERSION.equals(version)) {
            return condition.and(SCHEDULED_TASKS.VERSION.eq(version));
        }
        return condition;
    }

    @Override
    public Find find() {
        return new StorageFindImpl(this);
    }

    @Override
    public Update update() {
        return new StorageUpdateImpl(this);
    }

    /**
     * Добавляет новые задачи в бд, разархивирует задачи, если их снова надо запускать, архивирует старые задачи,
     * задачи, у которых поменяось расписание, помечаются, как требующие обновления расписания
     * 1) Если для всех строк бд версия расписания совпадает с текущей - ничего не надо делать
     * 2) Если есть отличная версия - надо обновить:
     * 3) Вставлет все новые записи раписания,
     * Если такая задача уже есть и ее расписание совпадает с раписанием новой, то у нее обновлется только версия
     * Все новые задачи и те, у которых отличается раписание, помечаются как need_reschedule становится = 1, у них
     * проставляется новая версия и новая хэш-сумма расписания
     * Если задача была архивная - она разархивируется
     * <p>
     * Вторым этапом архивируются задачи, у которых нет в новом расписании, у них тоже обновляется версия
     */
    @Override
    public void setNewSchedule(Collection<ScheduleRecord> newScheduledRecords) {
        if (allTasksHasStorageVersion()) {
            return;
        }

        var currentScheduleSet = getCurrentSchedule();

        Set<TaskId> newScheduleTaskIds =
                newScheduledRecords.stream().map(scheduledRecord -> new TaskIdImpl(scheduledRecord.getName(),
                        scheduledRecord.getParam())).collect(toSet());

        rescheduleTasks(newScheduledRecords);

        var archivedScheduleSet = currentScheduleSet.stream()
                .filter(job -> !newScheduleTaskIds.contains(job.taskId()))
                .map(el -> ((PrimaryIdImpl) el.primaryId()).getId())
                .collect(toSet());

        archiveTasks(archivedScheduleSet);
    }

    private boolean allTasksHasStorageVersion() {
        Set<String> taskVersions = ctxt.selectDistinct(SCHEDULED_TASKS.VERSION)
                .from(SCHEDULED_TASKS)
                .fetch()
                .intoSet(SCHEDULED_TASKS.VERSION);

        return taskVersions.contains(version) && taskVersions.size() == 1;
    }

    private Set<Job> getCurrentSchedule() {
        Field[] fields = SCHEDULED_TASKS.fields();

        return ctxt
                .select(fields)
                .from(SCHEDULED_TASKS)
                .fetchSet(jobRecordMapper);
    }


    private void rescheduleTasks(Collection<ScheduleRecord> scheduleRecords) {
        InsertSetStep<ScheduledTasksRecord> insertSetStep = ctxt
                .insertInto(SCHEDULED_TASKS);

        InsertSetMoreStep<ScheduledTasksRecord> insertSetMoreStep = null;

        for (var scheduledRecord : scheduleRecords) {
            ScheduledTasksRecord record = convertScheduleRecord(scheduledRecord);

            if (insertSetMoreStep != null) {
                insertSetStep = insertSetMoreStep.newRecord();
            }

            insertSetMoreStep = insertSetStep.set(record);
        }

        if (insertSetMoreStep != null) {
            insertSetMoreStep
                    .onDuplicateKeyUpdate()
                    .set(SCHEDULED_TASKS.NEED_RESCHEDULE,
                            DSL.decode()
                                    .when(SCHEDULED_TASKS.SCHEDULE_HASH
                                            .ne(MySQLDSL.values(SCHEDULED_TASKS.SCHEDULE_HASH)), 1L)
                                    .otherwise(SCHEDULED_TASKS.NEED_RESCHEDULE))
                    .set(SCHEDULED_TASKS.SCHEDULE_HASH, MySQLDSL.values(SCHEDULED_TASKS.SCHEDULE_HASH))
                    .set(SCHEDULED_TASKS.JOB_NAME_HASH, MySQLDSL.values(SCHEDULED_TASKS.JOB_NAME_HASH))
                    .set(SCHEDULED_TASKS.INSTANCE_ID,
                            DSL.when(SCHEDULED_TASKS.STATUS.ne(Deleted), SCHEDULED_TASKS.INSTANCE_ID)
                                    .otherwise((String) null))
                    .set(SCHEDULED_TASKS.STATUS,
                            DSL.when(SCHEDULED_TASKS.STATUS.ne(Deleted), SCHEDULED_TASKS.STATUS).otherwise(New))
                    .set(SCHEDULED_TASKS.VERSION, MySQLDSL.values(SCHEDULED_TASKS.VERSION))
                    .set(SCHEDULED_TASKS.META, MySQLDSL.values(SCHEDULED_TASKS.META))
                    .execute();
        }

    }

    private void archiveTasks(Collection<Long> ids) {
        if (!ids.isEmpty()) {
            ctxt.update(SCHEDULED_TASKS)
                    .set(SCHEDULED_TASKS.STATUS, Deleted)
                    .set(SCHEDULED_TASKS.VERSION, version)
                    .where(SCHEDULED_TASKS.ID.in(ids))
                    .execute();
        }
    }

    private ScheduledTasksRecord convertScheduleRecord(ScheduleRecord scheduleRecord) {
        ScheduledTasksRecord record = new ScheduledTasksRecord();

        record.setName(scheduleRecord.getName());
        record.setParams(scheduleRecord.getParam());
        record.setJobNameHash(scheduleRecord.getNameHashSum());
        record.setScheduleHash(scheduleRecord.getScheduleHashSum());
        record.setNeedReschedule(1L);
        record.setVersion(version);
        record.setMeta(scheduleRecord.getMeta());
        return record;
    }

    private Condition findRequestToCondition(StorageFindImpl find) {
        List<Condition> conditions = new ArrayList<>();

        if (find.getJobStatus() != null) {
            conditions.add(statusToCondition(find.getJobStatus()));
        }

        if (find.getPrimaryIds() != null) {
            List<Long> ids = find.getPrimaryIds().stream().map(t -> ((PrimaryIdImpl) t).getId()).collect(
                    toList());

            conditions.add(SCHEDULED_TASKS.ID.in(ids));
        }

        if (find.isNextRunLessThanNow()) {
            conditions.add(SCHEDULED_TASKS.NEXT_RUN.le(DSL.currentLocalDateTime()));
        }

        if (find.getNeedReschedule() != null) {
            conditions.add(SCHEDULED_TASKS.NEED_RESCHEDULE.eq(find.getNeedReschedule() ? 1L : 0L));
        }

        if (find.getTaskId() != null) {
            conditions.add(SCHEDULED_TASKS.NAME.eq(find.getTaskId().name()));
            conditions.add(SCHEDULED_TASKS.PARAMS.eq(find.getTaskId().param()));
        }

        return DSL.and(conditions); //return trueCondition, if conditions is empty
    }

    Collection<PrimaryId> findPrimaryId(StorageFindImpl find, int limit) {
        return ctxt
                .select(SCHEDULED_TASKS.ID)
                .from(SCHEDULED_TASKS)
                .where(
                        findRequestToCondition(find)
                )
                .limit(limit)
                .fetch(t -> new PrimaryIdImpl(t.value1()));
    }

    Collection<Job> findJobs(StorageFindImpl find, int limit) {
        Field[] fields = SCHEDULED_TASKS.fields();

        return ctxt
                .select(fields)
                .from(SCHEDULED_TASKS)
                .where(
                        findRequestToCondition(find)
                )
                .limit(limit)
                .fetch(jobRecordMapper);
    }

    public int executeUpdate(StorageUpdateImpl update) {
        ScheduledTasksRecord record = ctxt.newRecord(SCHEDULED_TASKS);

        var where = update.getFindCondition();

        if (update.getJobStatus() != null) {
            if (where.getJobStatus() != null) {
                if (!where.getJobStatus().getAllowedTransitions().contains(update.getJobStatus())) {
                    throw new IllegalArgumentException("Transition isn't allowed");
                }
            }

            setFieldsFromStatus(record, update.getJobStatus());
        }

        if (update.getNextRun() != null) {
            record.setNextRun(update.getNextRun());
        }

        if (update.getTaskId() != null) {
            record.setName(update.getTaskId().name());
            record.setParams(update.getTaskId().param());
        }

        if (update.getTaskProcessingResult() != null) {
            record.setLastStartTime(LocalDateTime.ofInstant(update.getTaskProcessingResult().lastStartTime(),
                    ZoneId.systemDefault()));
            record.setLastFinishTime(LocalDateTime.ofInstant(update.getTaskProcessingResult().lastFinishTime(),
                    ZoneId.systemDefault()));
        }

        if (update.getNeedReschedule() != null) {
            record.setNeedReschedule(update.getNeedReschedule() ? 1L : 0L);
        }

        return ctxt.update(SCHEDULED_TASKS)
                .set(record)
                .where(findRequestToCondition(where))
                .execute();
    }

    private static class JobRecordMapper implements RecordMapper<Record, Job> {

        @Override
        public Job map(Record record) {
            TaskId taskId = new TaskIdImpl(record.get(SCHEDULED_TASKS.NAME), record.get(SCHEDULED_TASKS.PARAMS));

            return new JobImpl(new PrimaryIdImpl(record.get(SCHEDULED_TASKS.ID)),
                    taskId,
                    record.get(SCHEDULED_TASKS.SCHEDULE_HASH),
                    TaskProcessingResultImpl.builder()
                            .withLastStartTime(getInstantFromLocalDateTime(record.get(SCHEDULED_TASKS.LAST_START_TIME)))
                            .withLastFinishTime(
                                    getInstantFromLocalDateTime(record.get(SCHEDULED_TASKS.LAST_FINISH_TIME)))
                            .build(),
                    getInstantFromLocalDateTime(record.get(SCHEDULED_TASKS.NEXT_RUN)),
                    parseJobStatus(record),
                    record.get(SCHEDULED_TASKS.NEED_RESCHEDULE) == 1L,
                    new InstanceIdImpl(record.get(SCHEDULED_TASKS.INSTANCE_ID)),
                    record.get(SCHEDULED_TASKS.META));
        }

        private Instant getInstantFromLocalDateTime(LocalDateTime localDateTime) {
            if (localDateTime == null) {
                return null;
            }
            return localDateTime.atZone(ZoneId.systemDefault()).toInstant();
        }

        private JobStatus parseJobStatus(Record record) {

            if (record.get(SCHEDULED_TASKS.STATUS).equals(New) && record.get(SCHEDULED_TASKS.INSTANCE_ID) == null) {
                return READY;
            } else if (record.get(SCHEDULED_TASKS.STATUS).equals(Paused)) {
                return STOPPED;
            } else if (record.get(SCHEDULED_TASKS.STATUS).equals(Deleted)) {
                return ARCHIVED;
            } else if (record.get(SCHEDULED_TASKS.INSTANCE_ID) != null) {
                return LOCKED;
            }

            return null;
        }
    }

}
