package ru.yandex.direct.hourglass.mysql.storage;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Collection;

import ru.yandex.direct.hourglass.TaskProcessingResult;
import ru.yandex.direct.hourglass.storage.JobStatus;
import ru.yandex.direct.hourglass.storage.PrimaryId;
import ru.yandex.direct.hourglass.storage.TaskId;
import ru.yandex.direct.hourglass.storage.Update;

public class StorageUpdateImpl implements Update {
    //Monday, 18 January 2038, 3:14:07, one day before the Mysql's timestamp end
    private static final int MAX_TIMESTAMP_EPOCH = 2147397247;

    private static final Instant MAX_DATE = Instant.ofEpochSecond(MAX_TIMESTAMP_EPOCH);

    private final StorageImpl storage;
    private final StorageFindImpl find;
    private LocalDateTime nextRun;
    private JobStatus jobStatus;
    private TaskProcessingResult taskProcessingResult;
    private TaskId taskId;
    private Boolean needReschedule = null;

    public StorageUpdateImpl(StorageImpl storage) {
        this.storage = storage;
        this.find = new StorageFindImpl(storage);
    }

    @Override
    public Update wherePrimaryIdIn(Collection<PrimaryId> primaryIds) {
        find.wherePrimaryIdIn(primaryIds);
        return this;
    }

    StorageFindImpl getFindCondition() {
        return find;
    }

    @Override
    public Update whereJobStatus(JobStatus jobStatus) {
        find.whereJobStatus(jobStatus);
        return this;
    }

    @Override
    public Update whereNextRunLeNow() {
        find.whereNextRunLeNow();
        return this;
    }

    @Override
    public Update whereTaskId(TaskId taskId) {
        find.whereTaskId(taskId);
        return this;
    }

    @Override
    public Update setNextRun(Instant nextRun) {
        Instant tmpNextRun;
        if (MAX_DATE.compareTo(nextRun) < 0) {
            tmpNextRun = MAX_DATE;
        } else {
            tmpNextRun = nextRun;
        }

        this.nextRun = LocalDateTime.ofInstant(tmpNextRun, ZoneOffset.systemDefault());
        return this;
    }

    @Override
    public Update setJobStatus(JobStatus jobStatus) {
        this.jobStatus = jobStatus;
        return this;
    }

    @Override
    public Update pingJob() {
        return this.setJobStatus(JobStatus.LOCKED);
    }

    @Override
    public Update setTaskProcessingResult(TaskProcessingResult taskProcessingResult) {
        this.taskProcessingResult = taskProcessingResult;
        return this;
    }

    @Override
    public Update setTaskId(TaskId taskId) {
        this.taskId = taskId;
        return this;
    }

    LocalDateTime getNextRun() {
        return nextRun;
    }

    JobStatus getJobStatus() {
        return jobStatus;
    }

    TaskProcessingResult getTaskProcessingResult() {
        return taskProcessingResult;
    }

    TaskId getTaskId() {
        return taskId;
    }

    Boolean getNeedReschedule() {
        return needReschedule;
    }

    @Override
    public int execute() {
        return storage.executeUpdate(this);
    }

    @Override
    public Update setNeedReschedule(boolean b) {
        this.needReschedule = b;
        return this;
    }

    @Override
    public Update whereNeedReschedule(boolean b) {
        find.whereNeedReschedule(b);
        return this;
    }
}
