package ru.yandex.direct.hourglass.mysql.schedulerinstances;

import java.sql.SQLException;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import org.assertj.core.data.TemporalUnitOffset;
import org.assertj.core.data.TemporalUnitWithinOffset;
import org.jooq.DSLContext;
import org.jooq.InsertValuesStepN;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import ru.yandex.direct.hourglass.InstanceId;
import ru.yandex.direct.hourglass.client.SchedulerInstance;
import ru.yandex.direct.hourglass.implementations.InstanceIdImpl;
import ru.yandex.direct.hourglass.implementations.updateschedule.SchedulerInstanceImpl;
import ru.yandex.direct.hourglass.mysql.DslContextHolder;
import ru.yandex.partner.dbschema.partner.tables.records.SchedulerInstancesRecord;

import static java.time.temporal.ChronoUnit.SECONDS;
import static java.util.stream.Collectors.toMap;
import static org.assertj.core.api.Assertions.assertThat;
import static ru.yandex.partner.dbschema.partner.Tables.SCHEDULER_INSTANCES;

class MysqlSchedulerInstancesRepositoryTest {

    private static final Duration HEARTBEAT_EXPIRATION = Duration.ofDays(120);
    private static final Duration INSTANCE_EXPIRATION = Duration.ofDays(1000);
    private static final TemporalUnitOffset OFFSET = new TemporalUnitWithinOffset(1, ChronoUnit.MINUTES);
    private static final LocalDateTime GOOD_HEARTBEAT_TIME = LocalDateTime.now().minusSeconds(15).truncatedTo(SECONDS);
    private static DSLContext dslContext;
    private MysqlSchedulerInstancesRepository mysqlSchedulerInstancesRepository;

    @BeforeAll
    static void initDb() throws SQLException, InterruptedException {
        dslContext = DslContextHolder.getDslContext();
    }

    @BeforeEach
    void before() {
        mysqlSchedulerInstancesRepository = new MysqlSchedulerInstancesRepository.Builder(dslContext)
                .withHeartbeatExpiration(HEARTBEAT_EXPIRATION)
                .withInstancesExpiration(INSTANCE_EXPIRATION)
                .build();

        clearRepository();
    }

    /**
     * Тест проверяет, что если неосновной инстанс помечен как основной - у него поменяется значение поля isMain
     * У остальных записей ничего не поменяется
     */
    @Test
    void markInstanceAsMainTest_isMainChanged() {
        var instanceIds = new ArrayList<InstanceId>();

        for (var i = 0; i < 3; i++) {
            instanceIds.add(new InstanceIdImpl());
        }

        var instanceInfoToBeChanged = new SchedulerInstancesInfo(instanceIds.get(1), GOOD_HEARTBEAT_TIME, 0, "13");
        var instancesInfoNotToBeChanged = List.of(
                new SchedulerInstancesInfo(instanceIds.get(0), GOOD_HEARTBEAT_TIME, 0, "12"),
                new SchedulerInstancesInfo(instanceIds.get(2), GOOD_HEARTBEAT_TIME, 1, "13"));
        insertSchedulerInstances(List.of(instanceInfoToBeChanged));

        insertSchedulerInstances(instancesInfoNotToBeChanged);

        mysqlSchedulerInstancesRepository.markInstanceAsMain(instanceInfoToBeChanged.instanceId);

        var schedulerInstancesInfoChangedGot =
                selectSchedulerInstances(List.of(instanceInfoToBeChanged.instanceId.toString()));

        assertThat(schedulerInstancesInfoChangedGot).hasSize(1);
        assertThat(schedulerInstancesInfoChangedGot.get(0).isMain).isEqualTo(1L);

        var schedulerInstancesInfoNotChangedGot =
                selectSchedulerInstances(instancesInfoNotToBeChanged.stream().map(i -> i.instanceId.toString())
                        .collect(Collectors.toList()));

        assertThat(schedulerInstancesInfoNotChangedGot).hasSize(2);
        assertThat(schedulerInstancesInfoNotChangedGot)
                .containsExactlyInAnyOrder(instancesInfoNotToBeChanged.toArray(SchedulerInstancesInfo[]::new));
    }

    /**
     * Тест проверяет, что если основной инстанс помечен как неосновной - у него поменяется значение поля isMain
     * У остальных записей ничего не поменяется
     */
    @Test
    void markInstanceAsNotMainTest() {
        var instanceIds = new ArrayList<InstanceId>();

        for (var i = 0; i < 3; i++) {
            instanceIds.add(new InstanceIdImpl());
        }

        var instanceInfoToBeChanged = new SchedulerInstancesInfo(instanceIds.get(1), GOOD_HEARTBEAT_TIME, 1, "13");
        var instancesInfoNotToBeChanged = List.of(
                new SchedulerInstancesInfo(instanceIds.get(0), GOOD_HEARTBEAT_TIME, 0, "12"),
                new SchedulerInstancesInfo(instanceIds.get(2), GOOD_HEARTBEAT_TIME, 1, "13"));
        insertSchedulerInstances(List.of(instanceInfoToBeChanged));

        insertSchedulerInstances(instancesInfoNotToBeChanged);

        mysqlSchedulerInstancesRepository.markInstanceAsNotMain(instanceInfoToBeChanged.instanceId);

        var schedulerInstancesInfoChangedGot =
                selectSchedulerInstances(List.of(instanceInfoToBeChanged.instanceId.toString()));

        assertThat(schedulerInstancesInfoChangedGot).hasSize(1);
        assertThat(schedulerInstancesInfoChangedGot.get(0).isMain).isEqualTo(0L);

        var schedulerInstancesInfoNotChangedGot =
                selectSchedulerInstances(instancesInfoNotToBeChanged.stream().map(i -> i.instanceId.toString())
                        .collect(Collectors.toList()));

        assertThat(schedulerInstancesInfoNotChangedGot).hasSize(2);
        assertThat(schedulerInstancesInfoNotChangedGot)
                .containsExactlyInAnyOrder(instancesInfoNotToBeChanged.toArray(SchedulerInstancesInfo[]::new));
    }

    /**
     * Тест проверяет, что если в таблице есть версия, у которой строк с полем isMain = 1 больше чем у всех
     * остальных, то она будет лидером, а остальные нет
     */
    @Test
    void isLeaderTest() {
        var instanceIds = new ArrayList<InstanceId>();

        for (var i = 0; i < 4; i++) {
            instanceIds.add(new InstanceIdImpl());
        }
        var leaderVersionCandidate = "13";
        var notLeaderVersionCandidate = "12";
        var instancesInfoToInsert = List.of(
                new SchedulerInstancesInfo(instanceIds.get(0), GOOD_HEARTBEAT_TIME, 1, leaderVersionCandidate),
                new SchedulerInstancesInfo(instanceIds.get(1), GOOD_HEARTBEAT_TIME, 1, notLeaderVersionCandidate),
                new SchedulerInstancesInfo(instanceIds.get(2), GOOD_HEARTBEAT_TIME, 1, leaderVersionCandidate),
                new SchedulerInstancesInfo(instanceIds.get(3), GOOD_HEARTBEAT_TIME, 1, leaderVersionCandidate));

        insertSchedulerInstances(instancesInfoToInsert);
        var isLeader = mysqlSchedulerInstancesRepository.isLeaderVersion(leaderVersionCandidate);
        var isNotLeader = mysqlSchedulerInstancesRepository.isLeaderVersion(notLeaderVersionCandidate);

        assertThat(isLeader).isTrue();
        assertThat(isNotLeader).isFalse();
    }

    /**
     * Тест проверяет, что если в таблице есть версия, у которой строк с полем isMain = 1 больше чем у всех
     * остальных, но это значение не превышает минимальный порог, то она не будет лидером
     */
    @Test
    void isLeader_LoaderBorderNotReachedTest() {
        var instanceIds = new ArrayList<InstanceId>();
        var repository =
                new MysqlSchedulerInstancesRepository.Builder(dslContext)
                .withHeartbeatExpiration(HEARTBEAT_EXPIRATION)
                .withInstancesExpiration(INSTANCE_EXPIRATION)
                .withLeaderVotingLowerBound(3)
                .build();

        for (var i = 0; i < 5; i++) {
            instanceIds.add(new InstanceIdImpl());
        }
        var leaderVersionCandidate = "13";
        var notLeaderVersionCandidate = "12";
        int i = 0;
        var instancesInfoToInsert = List.of(
                new SchedulerInstancesInfo(instanceIds.get(i++), GOOD_HEARTBEAT_TIME, 1, leaderVersionCandidate),
                new SchedulerInstancesInfo(instanceIds.get(i++), GOOD_HEARTBEAT_TIME, 1, leaderVersionCandidate),
                new SchedulerInstancesInfo(instanceIds.get(i++), GOOD_HEARTBEAT_TIME, 1, leaderVersionCandidate),
                new SchedulerInstancesInfo(instanceIds.get(i++), GOOD_HEARTBEAT_TIME, 1, notLeaderVersionCandidate),
                new SchedulerInstancesInfo(instanceIds.get(i), GOOD_HEARTBEAT_TIME, 1, leaderVersionCandidate));

        insertSchedulerInstances(instancesInfoToInsert);
        var isLeader = repository.isLeaderVersion(leaderVersionCandidate);
        var isNotLeader = repository.isLeaderVersion(notLeaderVersionCandidate);

        assertThat(isLeader).isTrue();
        assertThat(isNotLeader).isFalse();
    }

    /**
     * Тест проверяет, что если в таблице есть версия, у которой строк с полем isMain = 1 больше чем у всех
     * остальных и это значение превышает минимальный порог, то она будет лидером, а остальные нет
     */
    @Test
    void isLeader_LoaderBorderReachedTest() {
        var instanceIds = new ArrayList<InstanceId>();
        var repository = new MysqlSchedulerInstancesRepository.Builder(dslContext)
                .withHeartbeatExpiration(HEARTBEAT_EXPIRATION)
                .withInstancesExpiration(INSTANCE_EXPIRATION)
                .withLeaderVotingLowerBound(3)
                .build();

        for (var i = 0; i < 3; i++) {
            instanceIds.add(new InstanceIdImpl());
        }
        var leaderVersionCandidate = "13";
        var notLeaderVersionCandidate = "12";
        int i = 0;
        var instancesInfoToInsert = List.of(
                new SchedulerInstancesInfo(instanceIds.get(i++), GOOD_HEARTBEAT_TIME, 1, leaderVersionCandidate),
                new SchedulerInstancesInfo(instanceIds.get(i++), GOOD_HEARTBEAT_TIME, 1, notLeaderVersionCandidate),
                new SchedulerInstancesInfo(instanceIds.get(i), GOOD_HEARTBEAT_TIME, 1, leaderVersionCandidate));

        insertSchedulerInstances(instancesInfoToInsert);
        var isLeader = repository.isLeaderVersion(leaderVersionCandidate);
        var isNotLeader = repository.isLeaderVersion(notLeaderVersionCandidate);

        assertThat(isLeader).isFalse();
        assertThat(isNotLeader).isFalse();
    }

    /**
     * Тест проверяет, если версии нет в списке, то она не лидирующая
     */
    @Test
    void isLeaderTest_EmptyCandidates() {
        var isLeader = mysqlSchedulerInstancesRepository.isLeaderVersion("1");

        assertThat(isLeader).isFalse();
    }

    /**
     * Тест проверяет, что если две версии имеют одинкаовое количество строк с полем isMain=1, то ни она из них не
     * лидирующая
     */
    @Test
    void isLeaderTest_TwoPossibleCandidates() {
        var instanceIds = new ArrayList<InstanceId>();
        for (var i = 0; i < 4; i++) {
            instanceIds.add(new InstanceIdImpl());
        }
        var leaderVersionCandidate1 = "13";
        var leaderVersionCandidate2 = "12";
        var i = 0;
        var instancesInfoToInsert = List.of(
                new SchedulerInstancesInfo(instanceIds.get(i++), GOOD_HEARTBEAT_TIME, 1,
                        leaderVersionCandidate1),
                new SchedulerInstancesInfo(instanceIds.get(i++), GOOD_HEARTBEAT_TIME, 1, leaderVersionCandidate2),
                new SchedulerInstancesInfo(instanceIds.get(i++), GOOD_HEARTBEAT_TIME, 1, leaderVersionCandidate1),
                new SchedulerInstancesInfo(instanceIds.get(i), GOOD_HEARTBEAT_TIME, 1, leaderVersionCandidate2));

        insertSchedulerInstances(instancesInfoToInsert);
        var isLeader1 = mysqlSchedulerInstancesRepository.isLeaderVersion(leaderVersionCandidate1);
        var isLeader2 = mysqlSchedulerInstancesRepository.isLeaderVersion(leaderVersionCandidate2);

        assertThat(isLeader1).isFalse();
        assertThat(isLeader2).isFalse();
    }

    /**
     * Тест проверяет, строки, у которых поле isMain = true, но heartbeatTime очень старый, будут игнорироваться
     */
    @Test
    void isLeader_ExpiredHeartbeatIsSkippedTest() {
        var instanceIds = new ArrayList<InstanceId>();

        for (var i = 0; i < 5; i++) {
            instanceIds.add(new InstanceIdImpl());
        }

        var badHeartbeatTime = GOOD_HEARTBEAT_TIME.minus(HEARTBEAT_EXPIRATION);
        var leaderVersionCandidate = "13";
        var notLeaderVersionCandidate = "12";
        var instancesInfoToInsert = List.of(
                new SchedulerInstancesInfo(instanceIds.get(0), GOOD_HEARTBEAT_TIME, 1, leaderVersionCandidate),
                new SchedulerInstancesInfo(instanceIds.get(1), badHeartbeatTime, 1, notLeaderVersionCandidate),
                new SchedulerInstancesInfo(instanceIds.get(2), GOOD_HEARTBEAT_TIME, 1, leaderVersionCandidate),
                new SchedulerInstancesInfo(instanceIds.get(3), badHeartbeatTime, 1, notLeaderVersionCandidate),
                new SchedulerInstancesInfo(instanceIds.get(4), GOOD_HEARTBEAT_TIME, 1, notLeaderVersionCandidate));

        insertSchedulerInstances(instancesInfoToInsert);
        var isLeader = mysqlSchedulerInstancesRepository.isLeaderVersion(leaderVersionCandidate);
        var isNotLeader = mysqlSchedulerInstancesRepository.isLeaderVersion(notLeaderVersionCandidate);

        assertThat(isLeader).isTrue();
        assertThat(isNotLeader).isFalse();
    }

    /**
     * Тест проверяет, что строки с полем isMain=0 не участвуют в выборе лидера
     */
    @Test
    void isLeaderTest_OneIsNotMain() {
        var instanceIds = new ArrayList<InstanceId>();

        for (var i = 0; i < 4; i++) {
            instanceIds.add(new InstanceIdImpl());
        }
        var leaderVersionMain = "13";
        var leaderVersionNotMain = "12";
        var i = 0;
        var instancesInfoToInsert = List.of(
                new SchedulerInstancesInfo(instanceIds.get(i++), GOOD_HEARTBEAT_TIME, 1, leaderVersionMain),
                new SchedulerInstancesInfo(instanceIds.get(i++), GOOD_HEARTBEAT_TIME, 0, leaderVersionNotMain),
                new SchedulerInstancesInfo(instanceIds.get(i++), GOOD_HEARTBEAT_TIME, 1, leaderVersionMain),
                new SchedulerInstancesInfo(instanceIds.get(i), GOOD_HEARTBEAT_TIME, 1, leaderVersionNotMain));

        insertSchedulerInstances(instancesInfoToInsert);
        var isLeader1 = mysqlSchedulerInstancesRepository.isLeaderVersion(leaderVersionMain);
        var isLeader2 = mysqlSchedulerInstancesRepository.isLeaderVersion(leaderVersionNotMain);

        assertThat(isLeader1).isTrue();
        assertThat(isLeader2).isFalse();
    }

    /**
     * Тест проверяет, что поле isActive заполняется верно
     */
    @Test
    void IsActiveTest() {
        var instanceIds = new ArrayList<InstanceId>();

        for (var i = 0; i < 2; i++) {
            instanceIds.add(new InstanceIdImpl(i + ""));
        }

        var i = 0;

        var oldHeartbeat = LocalDateTime.now().minus(INSTANCE_EXPIRATION);
        var freshHeartbeat = LocalDateTime.now().plusDays(1);

        var instancesInfoToInsert = List.of(
                new SchedulerInstancesInfo(instanceIds.get(i++), oldHeartbeat, 1, ""),
                new SchedulerInstancesInfo(instanceIds.get(i), freshHeartbeat, 1, ""));

        insertSchedulerInstances(instancesInfoToInsert);

        List<SchedulerInstance> schedulerInstances = mysqlSchedulerInstancesRepository.getSchedulerInstancesInfo();

        assertThat(schedulerInstances).hasSize(2);

        var firstInstance =
                schedulerInstances.stream().filter(e -> e.getInstanceId().toString().equals("0")).findFirst();
        var secondInstance =
                schedulerInstances.stream().filter(e -> e.getInstanceId().toString().equals("1")).findFirst();

        assertThat(firstInstance).isPresent();
        assertThat(secondInstance).isPresent();

        assertThat(firstInstance.orElseThrow().isActive()).isFalse();
        assertThat(secondInstance.orElseThrow().isActive()).isTrue();
    }


    /**
     * Тест проверяет, что если есть протухшие инстансы, они удалятся при выборе лидера
     */
    @Test
    void isLeaderTest_ExpiredInstancesRemovedTest() {
        var instanceIds = new ArrayList<InstanceId>();

        for (var i = 0; i < 2; i++) {
            instanceIds.add(new InstanceIdImpl());
        }
        var i = 0;
        var heartbeatTimeToDeleteInstance = LocalDateTime.now().minus(INSTANCE_EXPIRATION).minusDays(1);
        var heartbeatTimeToStayInstance = LocalDateTime.now().minus(INSTANCE_EXPIRATION).plusDays(1);
        var instancesInfoToInsert = List.of(
                new SchedulerInstancesInfo(instanceIds.get(i++), heartbeatTimeToDeleteInstance, 1, ""),
                new SchedulerInstancesInfo(instanceIds.get(i), heartbeatTimeToStayInstance, 1, ""));

        insertSchedulerInstances(instancesInfoToInsert);
        mysqlSchedulerInstancesRepository.isLeaderVersion("");

        var gotSchedulerInstancesInfo = selectSchedulerInstances(List.of(instanceIds.get(0).toString(),
                instanceIds.get(1).toString()));

        assertThat(gotSchedulerInstancesInfo).hasSize(1);
        assertThat(gotSchedulerInstancesInfo.get(0).instanceId).isEqualTo(instanceIds.get(1));
    }

    @Test
    void getSchedulerInstancesInfoTest() {
        var instanceIds = new ArrayList<InstanceId>();
        var now = LocalDateTime.now().truncatedTo(SECONDS);
        for (var i = 0; i < 2; i++) {
            instanceIds.add(new InstanceIdImpl());
        }
        var i = 0;
        var heartbeatTime1 = now.minusMinutes(1);
        var heartbeatTime2 = now.minusMinutes(2);
        var instancesInfoToInsert = List.of(
                new SchedulerInstancesInfo(instanceIds.get(i++), heartbeatTime1, 1, "1", "meta1"),
                new SchedulerInstancesInfo(instanceIds.get(i), heartbeatTime2, 0, "5", "meta2"));

        insertSchedulerInstances(instancesInfoToInsert);
        var gotSchedulerInstances = mysqlSchedulerInstancesRepository.getSchedulerInstancesInfo();

        var expectedSchedulerInstances = new SchedulerInstance[]{
                new SchedulerInstanceImpl(instanceIds.get(0), "1", true,
                        heartbeatTime1.atZone(ZoneId.systemDefault()).toInstant(),
                        "meta1", true),
                new SchedulerInstanceImpl(instanceIds.get(1), "5", false,
                        heartbeatTime2.atZone(ZoneId.systemDefault()).toInstant(),
                        "meta2", true)
        };

        assertThat(gotSchedulerInstances).hasSize(2);
        assertThat(gotSchedulerInstances).containsExactlyInAnyOrder(expectedSchedulerInstances);
    }

    @Test
    void getPingInstanceTest() {
        var instanceIds = new ArrayList<InstanceId>();
        var now = LocalDateTime.now().truncatedTo(SECONDS);
        for (var i = 0; i < 2; i++) {
            instanceIds.add(new InstanceIdImpl());
        }
        var i = 0;
        var heartbeatTime1 = now.minusDays(2);
        var heartbeatTime2 = now.minusDays(1);
        var instancesInfoToInsert = List.of(
                new SchedulerInstancesInfo(instanceIds.get(i++), heartbeatTime1, 1, "1", "meta1"),
                new SchedulerInstancesInfo(instanceIds.get(i), heartbeatTime2, 0, "5", "meta2"));

        insertSchedulerInstances(instancesInfoToInsert);
        mysqlSchedulerInstancesRepository.pingInstance(instanceIds.get(0));

        var schedulerInstanceIdToInfo = selectSchedulerInstances(List.of(instanceIds.get(0).toString(),
                instanceIds.get(1).toString()))
                .stream()
                .collect(toMap(schedulerInstanceInfo -> schedulerInstanceInfo.instanceId,
                        schedulerInstanceInfo -> schedulerInstanceInfo));

        var changedSchedulerInstanceInfo = schedulerInstanceIdToInfo.get(instanceIds.get(0));
        assertThat(changedSchedulerInstanceInfo.heartbeatTime).isCloseTo(now, OFFSET);
        assertThat(changedSchedulerInstanceInfo.isMain).isEqualTo(1L);
        assertThat(changedSchedulerInstanceInfo.version).isEqualTo("1");
        assertThat(changedSchedulerInstanceInfo.meta).isEqualTo("meta1");
        assertThat(schedulerInstanceIdToInfo.get(instanceIds.get(1))).isEqualTo(instancesInfoToInsert.get(1));
    }

    private void insertSchedulerInstances(List<SchedulerInstancesInfo> schedulerInstancesInfoList) {
        InsertValuesStepN<SchedulerInstancesRecord> insertValuesStep = dslContext.insertInto(SCHEDULER_INSTANCES,
                List.of(SCHEDULER_INSTANCES.INSTANCE_ID, SCHEDULER_INSTANCES.HEARTBEAT_TIME,
                        SCHEDULER_INSTANCES.IS_MAIN, SCHEDULER_INSTANCES.VERSION, SCHEDULER_INSTANCES.META));

        schedulerInstancesInfoList.forEach(
                schedulerInstancesInfo -> insertValuesStep.values(schedulerInstancesInfo.instanceId,
                        schedulerInstancesInfo.heartbeatTime, schedulerInstancesInfo.isMain,
                        schedulerInstancesInfo.version, schedulerInstancesInfo.meta)
        );
        insertValuesStep.execute();
    }

    private List<SchedulerInstancesInfo> selectSchedulerInstances(List<String> instanceId) {
        return dslContext.select(SCHEDULER_INSTANCES.INSTANCE_ID,
                SCHEDULER_INSTANCES.HEARTBEAT_TIME,
                SCHEDULER_INSTANCES.IS_MAIN, SCHEDULER_INSTANCES.VERSION, SCHEDULER_INSTANCES.META)
                .from(SCHEDULER_INSTANCES)
                .where(SCHEDULER_INSTANCES.INSTANCE_ID.in(instanceId))
                .fetch(r ->
                        new SchedulerInstancesInfo(new InstanceIdImpl(r.get(SCHEDULER_INSTANCES.INSTANCE_ID)),
                                r.get(SCHEDULER_INSTANCES.HEARTBEAT_TIME), r.get(SCHEDULER_INSTANCES.IS_MAIN),
                                r.get(SCHEDULER_INSTANCES.VERSION), r.get(SCHEDULER_INSTANCES.META)));

    }

    private void clearRepository() {
        dslContext.deleteFrom(SCHEDULER_INSTANCES).execute();

    }

    private class SchedulerInstancesInfo {
        InstanceId instanceId;
        LocalDateTime heartbeatTime;
        long isMain;
        String version;
        String meta;

        SchedulerInstancesInfo(InstanceId instanceId, LocalDateTime heartbeatTime, long isMain, String version) {
            this(instanceId, heartbeatTime, isMain, version, "");
        }

        SchedulerInstancesInfo(InstanceId instanceId, LocalDateTime heartbeatTime, long isMain, String version,
                               String meta) {
            this.instanceId = instanceId;
            this.heartbeatTime = heartbeatTime;
            this.isMain = isMain;
            this.version = version;
            this.meta = meta;
        }


        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            SchedulerInstancesInfo that = (SchedulerInstancesInfo) o;
            return isMain == that.isMain &&
                    Objects.equals(instanceId, that.instanceId) &&
                    Objects.equals(heartbeatTime, that.heartbeatTime) &&
                    Objects.equals(version, that.version) &&
                    Objects.equals(meta, that.meta);
        }

        @Override
        public int hashCode() {
            return Objects.hash(instanceId, heartbeatTime, isMain, version, meta);
        }

        @Override
        public String toString() {
            return "SchedulerInstancesInfo{" +
                    "instanceId=" + instanceId +
                    ", heartbeatTime=" + heartbeatTime +
                    ", isMain=" + isMain +
                    ", version='" + version + '\'' +
                    ", meta='" + meta + '\'' +
                    '}';
        }
    }
}
