package ru.yandex.direct.hourglass.mysql.storage;

import java.sql.SQLException;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.Result;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import ru.yandex.direct.hourglass.HourglassProperties;
import ru.yandex.direct.hourglass.InstanceId;
import ru.yandex.direct.hourglass.mysql.DslContextHolder;
import ru.yandex.direct.hourglass.storage.Job;
import ru.yandex.direct.hourglass.storage.JobStatus;
import ru.yandex.direct.hourglass.storage.PrimaryId;

import static java.time.temporal.ChronoUnit.SECONDS;
import static java.util.stream.Collectors.toList;
import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static ru.yandex.partner.dbschema.partner.Tables.SCHEDULED_TASKS;
import static ru.yandex.partner.dbschema.partner.enums.ScheduledTasksStatus.New;
import static ru.yandex.partner.dbschema.partner.enums.ScheduledTasksStatus.Paused;
import static ru.yandex.partner.dbschema.partner.enums.ScheduledTasksStatus.Running;

class ExecutingJobsTest {
    static DSLContext dslContext;
    static StorageImpl storage;
    static InstanceId schedulerId;

    @BeforeAll
    static void initDb() throws SQLException, InterruptedException {


        dslContext = DslContextHolder.getDslContext();
        schedulerId = mock(InstanceId.class);

        when(schedulerId.toString()).thenReturn("deadbeef");

        storage = new StorageImpl(dslContext, schedulerId, HourglassProperties.builder().build(), "1");
    }

    @BeforeEach
    void makeSchedule() {
        dslContext.truncate(SCHEDULED_TASKS).execute();

        dslContext.insertInto(SCHEDULED_TASKS,
                SCHEDULED_TASKS.ID, SCHEDULED_TASKS.NAME, SCHEDULED_TASKS.PARAMS,
                SCHEDULED_TASKS.STATUS, SCHEDULED_TASKS.NEED_RESCHEDULE, SCHEDULED_TASKS.HEARTBEAT_TIME,
                SCHEDULED_TASKS.NEXT_RUN,
                SCHEDULED_TASKS.SCHEDULE_HASH, SCHEDULED_TASKS.JOB_NAME_HASH,
                SCHEDULED_TASKS.INSTANCE_ID, SCHEDULED_TASKS.VERSION)


                .values(1L, "A", "A1", Running, 0L, LocalDateTime.now().minusSeconds(10),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        "deadbeef", "1")

                /* Во время выполнения поменялась версия расписания, по окончании работы версия не должна сброситься */
                .values(2L, "B", "B1", Running, 1L, LocalDateTime.now().minusSeconds(10),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        "deadbeef", "2")


                //Not executing jobs
                .values(3L, "C", "B1", Running, 0L, LocalDateTime.now().minusSeconds(10),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        "cafebabe", "2")
                .values(4L, "D", "B1", New, 0L, null,
                        LocalDateTime.now().minusSeconds(10), "", "",
                        null, "1")
                .values(5L, "E", "B1", Paused, 0L, null,
                        LocalDateTime.now().minusSeconds(10), "", "",
                        null, "2")
                .values(6L, "F", "B1", New, 1L, null,
                        LocalDateTime.now().minusSeconds(10), "", "",
                        "deadbeef", "2")

                .execute();
    }

    @Test
    void findTest() {

        Collection<PrimaryId> primaryIds = storage.find().whereJobStatus(JobStatus.LOCKED).findPrimaryIds();
        Collection<Job> jobs =
                storage.find().wherePrimaryIdIn(primaryIds).whereJobStatus(JobStatus.LOCKED).findJobs();

        Set<Long> expectedIds = new HashSet<>(Set.of(1L, 2L));

        assertThat(jobs).hasSize(2);
        var gotIds = jobs.stream().map(job -> ((PrimaryIdImpl) job.primaryId()).getId()).collect(toList());
        assertThat(gotIds).containsExactlyInAnyOrder(expectedIds.toArray(Long[]::new));
    }

    @Test
    void updateTest() {
        LocalDateTime now = LocalDateTime.now().minusSeconds(1);
        storage.update().whereJobStatus(JobStatus.LOCKED).setJobStatus(JobStatus.LOCKED).execute();

        Result<? extends Record> result =
                dslContext.select(SCHEDULED_TASKS.ID, SCHEDULED_TASKS.HEARTBEAT_TIME)
                        .from(SCHEDULED_TASKS)
                        .where(SCHEDULED_TASKS.STATUS.eq(Running)
                                .and(SCHEDULED_TASKS.INSTANCE_ID.eq("deadbeef")))
                        .fetch();

        assertThat(result).hasSize(2);

        var expectedIds = List.of(1L, 2L);
        var gotIds = result.map(r -> r.get(SCHEDULED_TASKS.ID));
        assertThat(gotIds).containsExactlyInAnyOrder(expectedIds.toArray(Long[]::new));
        var heartbeatTimes = result.map(r -> r.get(SCHEDULED_TASKS.HEARTBEAT_TIME));

        for (var heartbeatTime : heartbeatTimes) {
            assertThat(heartbeatTime).isAfterOrEqualTo(now);
        }
    }

    @Test
    void executingToReadyTest() {
        Instant nextRun = Instant.now().truncatedTo(SECONDS);
        storage.update()
                .whereJobStatus(JobStatus.LOCKED)
                .setNextRun(nextRun)
                .setJobStatus(JobStatus.READY)
                .execute();

        List<RescheduledInfo> got =
                dslContext.select(SCHEDULED_TASKS.ID, SCHEDULED_TASKS.NEED_RESCHEDULE, SCHEDULED_TASKS.VERSION)
                        .from(SCHEDULED_TASKS)
                        .where(SCHEDULED_TASKS.STATUS.eq(New)
                                .and(SCHEDULED_TASKS.HEARTBEAT_TIME.isNull())
                                .and(SCHEDULED_TASKS.NEXT_RUN.eq(LocalDateTime.ofInstant(nextRun,
                                        ZoneId.systemDefault())))
                                .and(SCHEDULED_TASKS.INSTANCE_ID.isNull()))
                        .fetch(r -> new RescheduledInfo(r.get(SCHEDULED_TASKS.ID),
                                r.get(SCHEDULED_TASKS.NEED_RESCHEDULE), r.get(SCHEDULED_TASKS.VERSION)));

        assertThat(got).hasSize(2);
        var expected = new RescheduledInfo[]{
                new RescheduledInfo(1L, 0L, "1"),
                new RescheduledInfo(2L, 1L, "2")
        };
        assertThat(got).containsExactlyInAnyOrder(expected);
    }

    private class RescheduledInfo {
        long id;
        long needReschedule;
        String version;

        RescheduledInfo(long id, long needReschedule, String version) {
            this.id = id;
            this.needReschedule = needReschedule;
            this.version = version;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            RescheduledInfo that = (RescheduledInfo) o;
            return id == that.id &&
                    needReschedule == that.needReschedule &&
                    Objects.equals(version, that.version);
        }

        @Override
        public int hashCode() {
            return Objects.hash(id, needReschedule, version);
        }

        @Override
        public String toString() {
            return "RescheduledInfo{" +
                    "id=" + id +
                    ", needReschedule=" + needReschedule +
                    ", version='" + version + '\'' +
                    '}';
        }
    }

}
