package ru.yandex.direct.hourglass.mysql.storage;

import java.sql.SQLException;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.jooq.DSLContext;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import ru.yandex.direct.hourglass.HourglassProperties;
import ru.yandex.direct.hourglass.InstanceId;
import ru.yandex.direct.hourglass.mysql.DslContextHolder;
import ru.yandex.direct.hourglass.storage.Job;
import ru.yandex.direct.hourglass.storage.JobStatus;
import ru.yandex.direct.hourglass.storage.PrimaryId;
import ru.yandex.partner.dbschema.partner.enums.ScheduledTasksStatus;

import static java.time.temporal.ChronoUnit.SECONDS;
import static java.util.stream.Collectors.toList;
import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static ru.yandex.partner.dbschema.partner.Tables.SCHEDULED_TASKS;
import static ru.yandex.partner.dbschema.partner.enums.ScheduledTasksStatus.New;

class ExpiredJobsTest {
    static DSLContext dslContext;
    static StorageImpl storage;
    static InstanceId schedulerId;

    @BeforeAll
    static void initDb() throws SQLException, InterruptedException {


        dslContext = DslContextHolder.getDslContext();
        schedulerId = mock(InstanceId.class);

        when(schedulerId.toString()).thenReturn("deadbeef");
        var hourglassConfiguration = HourglassProperties.builder().setMaxHeartbeatAge(60, SECONDS).build();
        storage = new StorageImpl(dslContext, schedulerId, hourglassConfiguration, "1");
    }

    @BeforeEach
    void makeSchedule() {
        dslContext.truncate(SCHEDULED_TASKS).execute();

        dslContext.insertInto(SCHEDULED_TASKS,
                SCHEDULED_TASKS.ID, SCHEDULED_TASKS.NAME, SCHEDULED_TASKS.PARAMS,
                SCHEDULED_TASKS.STATUS, SCHEDULED_TASKS.NEED_RESCHEDULE, SCHEDULED_TASKS.HEARTBEAT_TIME,
                SCHEDULED_TASKS.NEXT_RUN,
                SCHEDULED_TASKS.SCHEDULE_HASH, SCHEDULED_TASKS.JOB_NAME_HASH,
                SCHEDULED_TASKS.INSTANCE_ID, SCHEDULED_TASKS.VERSION)

                //Expired
                .values(1L, "A", "A1", New, 0L, LocalDateTime.now().minusSeconds(600),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        "deadbeef", "1")
                .values(2L, "B", "B1", New, 1L, LocalDateTime.now().minusSeconds(600),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        "deadbeef", "2")
                .values(3L, "BG", "B1", New, 1L, LocalDateTime.now().minusSeconds(600),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        "cafebabe", "1")
                .values(4L, "C", "C1", ScheduledTasksStatus.Running, 0L, LocalDateTime.now().minusSeconds(600),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        "deadbeef", "2")
                .values(5L, "D", "C1", ScheduledTasksStatus.Running, 1L, LocalDateTime.now().minusSeconds(600),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        "deadbeef", "1")
                .values(6L, "DG", "C1", ScheduledTasksStatus.Running, 1L, LocalDateTime.now().minusSeconds(600),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        "cafebabe", "2")

                //Non expired
                .values(7L, "F", "F1", ScheduledTasksStatus.Paused, 1L, LocalDateTime.now().minusSeconds(1),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        "deadbeef", "1")
                .values(8L, "E", "E1", ScheduledTasksStatus.Paused, 0L, LocalDateTime.now().minusSeconds(1),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        "deadbeef", "2")
                .values(9L, "EG", "E1", ScheduledTasksStatus.Paused, 0L, LocalDateTime.now().minusSeconds(1),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        "cafebabe", "1")

                .values(10L, "J", "F1", ScheduledTasksStatus.Deleted, 1L, LocalDateTime.now().minusSeconds(1),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        "deadbeef", "1")
                .values(11L, "K", "E1", ScheduledTasksStatus.Deleted, 0L, LocalDateTime.now().minusSeconds(1),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        "deadbeef", "2")

                .values(12L, "L", "D1", New, 1L, LocalDateTime.now().minusSeconds(1),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        "deadbeef", "2")
                .values(13L, "LG", "D1", New, 1L, LocalDateTime.now().minusSeconds(1),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        "cafebabe", "1")
                .values(14L, "M", "F1", ScheduledTasksStatus.Running, 1L, LocalDateTime.now().minusSeconds(1),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        "deadbeef", "1")
                .values(15L, "MG", "F1", ScheduledTasksStatus.Running, 1L, LocalDateTime.now().minusSeconds(1),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        "cafebabe", "2")
                .execute();
    }

    @Test
    void findTest() {

        Collection<PrimaryId> primaryIds = storage.find().whereJobStatus(JobStatus.EXPIRED).findPrimaryIds();
        Collection<Job> jobs = storage.find().wherePrimaryIdIn(primaryIds).whereJobStatus(JobStatus.EXPIRED).findJobs();

        Set<Long> expectedExpiredIds = new HashSet<>(Set.of(4L, 5L, 6L));

        assertThat(jobs).hasSize(3);
        var gotExpiredIds = jobs.stream().map(job -> ((PrimaryIdImpl) job.primaryId()).getId()).collect(toList());
        assertThat(gotExpiredIds).containsExactlyInAnyOrder(expectedExpiredIds.toArray(Long[]::new));

    }

    @Test
    void updateTest() {
        storage.update().whereJobStatus(JobStatus.EXPIRED).setJobStatus(JobStatus.READY).execute();

        var gotIdsToVersions =
                dslContext.select(SCHEDULED_TASKS.ID, SCHEDULED_TASKS.VERSION)
                        .from(SCHEDULED_TASKS)
                        .where(SCHEDULED_TASKS.STATUS.eq(New)
                                .and(SCHEDULED_TASKS.INSTANCE_ID.isNull()
                                        .and(SCHEDULED_TASKS.HEARTBEAT_TIME.isNull())))
                        .fetchMap(r -> r.get(SCHEDULED_TASKS.ID), r -> r.get(SCHEDULED_TASKS.VERSION));

        assertThat(gotIdsToVersions).hasSize(3);
        var expectedIdsToVersions = Map.of(
                4L, "2",
                5L, "1",
                6L, "2"
        );

        assertThat(gotIdsToVersions).isEqualTo(expectedIdsToVersions);
    }
}
