package ru.yandex.direct.hourglass.mysql.storage;

import java.sql.SQLException;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Collection;
import java.util.Collections;

import org.jooq.DSLContext;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import ru.yandex.direct.hourglass.HourglassProperties;
import ru.yandex.direct.hourglass.InstanceId;
import ru.yandex.direct.hourglass.mysql.DslContextHolder;
import ru.yandex.direct.hourglass.storage.Job;
import ru.yandex.direct.hourglass.storage.JobStatus;
import ru.yandex.direct.hourglass.storage.PrimaryId;

import static java.time.temporal.ChronoUnit.HOURS;
import static java.time.temporal.ChronoUnit.SECONDS;
import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static ru.yandex.direct.hourglass.storage.JobStatus.READY;
import static ru.yandex.partner.dbschema.partner.Tables.SCHEDULED_TASKS;
import static ru.yandex.partner.dbschema.partner.enums.ScheduledTasksStatus.New;
import static ru.yandex.partner.dbschema.partner.enums.ScheduledTasksStatus.Running;

class NeedRescheduleTest {
    private static DSLContext dslContext;
    private static StorageImpl storage;
    private static InstanceId schedulerId;

    @BeforeAll
    static void initDb() throws SQLException, InterruptedException {
        dslContext = DslContextHolder.getDslContext();
        schedulerId = mock(InstanceId.class);

        when(schedulerId.toString()).thenReturn("deadbeef");
        var hourglassConfiguration = HourglassProperties.builder().setMaxHeartbeatAge(60, SECONDS).build();

        storage = new StorageImpl(dslContext, schedulerId, hourglassConfiguration, "1");
    }

    @BeforeEach
    void makeSchedule() {
        dslContext.truncate(SCHEDULED_TASKS).execute();

        dslContext.insertInto(SCHEDULED_TASKS,
                SCHEDULED_TASKS.ID, SCHEDULED_TASKS.NAME, SCHEDULED_TASKS.PARAMS,
                SCHEDULED_TASKS.STATUS, SCHEDULED_TASKS.NEED_RESCHEDULE, SCHEDULED_TASKS.HEARTBEAT_TIME,
                SCHEDULED_TASKS.NEXT_RUN,
                SCHEDULED_TASKS.SCHEDULE_HASH, SCHEDULED_TASKS.JOB_NAME_HASH,
                SCHEDULED_TASKS.INSTANCE_ID, SCHEDULED_TASKS.VERSION)

                .values(1L, "A", "A1", New, 1L, LocalDateTime.now().minusSeconds(600),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        null, "1")
                .values(2L, "B", "B1", Running, 1L, LocalDateTime.now().minusSeconds(600),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        "cafebabe", "1")
                .values(3L, "BG", "B1", New, 0L, LocalDateTime.now().minusSeconds(600),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        null, "1")
                .values(4L, "C", "C1", New, 1L, LocalDateTime.now().minusSeconds(600),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        "cafebabe", "2")
                .values(5L, "D", "D1", New, 1L, LocalDateTime.now().minusSeconds(600),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        "deadbeef", "2")
                .values(6L, "ANOTHER_VERSION_TASK", "A1", New, 1L, LocalDateTime.now().minusSeconds(600),
                        LocalDateTime.now().minusSeconds(10), "", "",
                        null, "2")
                .execute();
    }

    @Test
    void findTest() {

        Collection<PrimaryId> primaryIds = storage.find()
                .whereJobStatus(JobStatus.READY)
                .whereNeedReschedule(true)
                .findPrimaryIds();

        Collection<Job> jobs =
                storage.find().wherePrimaryIdIn(primaryIds)
                        .whereJobStatus(JobStatus.READY)
                        .whereNeedReschedule(true)
                        .findJobs();

        assertThat(jobs).hasSize(1);
        assertThat(((PrimaryIdImpl) jobs.iterator().next().primaryId()).getId()).isEqualTo(1L);
    }

    @Test
    void updateTest() {
        Instant nextRun = Instant.now().plus(1, HOURS).truncatedTo(SECONDS);

        storage.update()
                .wherePrimaryIdIn(Collections.singleton(new PrimaryIdImpl(1L)))
                .whereJobStatus(READY)
                .whereNeedReschedule(true)
                .setNextRun(nextRun)
                .setNeedReschedule(false)
                .execute();

        var gotIds = dslContext.select(SCHEDULED_TASKS.ID)
                .from(SCHEDULED_TASKS)
                .where(SCHEDULED_TASKS.STATUS.eq(New)
                        .and(SCHEDULED_TASKS.NEED_RESCHEDULE.eq(0L)
                                .and(SCHEDULED_TASKS.NEXT_RUN.eq(LocalDateTime.ofInstant(nextRun,
                                        ZoneId.systemDefault()))
                                        .and(SCHEDULED_TASKS.INSTANCE_ID.isNull()))))
                .fetch(SCHEDULED_TASKS.ID);

        assertThat(gotIds).hasSize(1);
        assertThat(gotIds.get(0)).isEqualTo(1L);
    }
}
