package ru.yandex.direct.hourglass.mysql.storage;

import java.sql.SQLException;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.assertj.core.data.TemporalUnitWithinOffset;
import org.jooq.DSLContext;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import ru.yandex.direct.hourglass.HourglassProperties;
import ru.yandex.direct.hourglass.InstanceId;
import ru.yandex.direct.hourglass.mysql.DslContextHolder;
import ru.yandex.direct.hourglass.storage.Job;
import ru.yandex.direct.hourglass.storage.JobStatus;
import ru.yandex.direct.hourglass.storage.PrimaryId;

import static java.util.stream.Collectors.toList;
import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static ru.yandex.partner.dbschema.partner.Tables.SCHEDULED_TASKS;
import static ru.yandex.partner.dbschema.partner.enums.ScheduledTasksStatus.New;
import static ru.yandex.partner.dbschema.partner.enums.ScheduledTasksStatus.Paused;
import static ru.yandex.partner.dbschema.partner.enums.ScheduledTasksStatus.Running;

class ReadyJobsTest {
    private static final String VERSION = "1";
    private static final LocalDateTime NOW = LocalDateTime.now();
    private static DSLContext dslContext;
    private static StorageImpl storage;
    private static InstanceId schedulerId;

    @BeforeAll
    static void initDb() throws SQLException, InterruptedException {


        dslContext = DslContextHolder.getDslContext();
        schedulerId = mock(InstanceId.class);

        when(schedulerId.toString()).thenReturn("deadbeef");

        storage = new StorageImpl(dslContext, schedulerId, HourglassProperties.builder().build(), VERSION);
    }

    @BeforeEach
    void makeSchedule() {
        dslContext.truncate(SCHEDULED_TASKS).execute();

        dslContext.insertInto(SCHEDULED_TASKS,
                SCHEDULED_TASKS.ID, SCHEDULED_TASKS.NAME, SCHEDULED_TASKS.PARAMS,
                SCHEDULED_TASKS.STATUS, SCHEDULED_TASKS.NEED_RESCHEDULE, SCHEDULED_TASKS.HEARTBEAT_TIME,
                SCHEDULED_TASKS.NEXT_RUN,
                SCHEDULED_TASKS.SCHEDULE_HASH, SCHEDULED_TASKS.JOB_NAME_HASH,
                SCHEDULED_TASKS.INSTANCE_ID,
                SCHEDULED_TASKS.VERSION)

                .values(1L, "A", "A1", New, 0L, LocalDateTime.now(), NOW.minusSeconds(10), "", "",
                        null, VERSION)

                .values(2L, "B", "A1", New, 0L, LocalDateTime.now(), NOW.minusSeconds(1), "", "",
                        null, VERSION)

                .values(3L, "C", "A1", New, 1L, LocalDateTime.now(), NOW.minusSeconds(1), "", "",
                        null, VERSION)

                .values(4L, "D", "A1", New, 0L, LocalDateTime.now(), NOW.minusSeconds(10), "", "",
                        "deadbeef", VERSION)

                .values(5L, "E", "A1", New, 0L, LocalDateTime.now(), NOW.minusSeconds(10), "", "",
                        "cafrbabe", VERSION)

                .values(6L, "F", "A1", New, 0L, LocalDateTime.now(), NOW.plusHours(1), "", "",
                        null, VERSION)

                .values(7L, "G", "A1", Paused, 0L, LocalDateTime.now(), NOW.minusMinutes(1), "", "",
                        null, VERSION)

                .values(8L, "J", "A1", Running, 0L, NOW.minusMinutes(2), NOW.minusSeconds(10), "", "",
                        "deadbeef", VERSION)

                .values(9L, "M", "P1", New, 0L, LocalDateTime.now(), NOW.minusSeconds(10), "", "",
                        null, "2")

                .execute();
    }

    @Test
    void findTest() {

        Collection<PrimaryId> primaryIds =
                storage.find().whereNextRunLeNow().whereJobStatus(JobStatus.READY).findPrimaryIds();

        Collection<Job> jobs = storage.find().wherePrimaryIdIn(primaryIds).whereJobStatus(JobStatus.READY).findJobs();

        assertThat(jobs).hasSize(3);
        var expectedIds = List.of(1L, 2L, 3L);
        var gotIds = jobs.stream().map(job -> ((PrimaryIdImpl) job.primaryId()).getId()).collect(toList());

        assertThat(gotIds).containsExactlyInAnyOrder(expectedIds.toArray(Long[]::new));
    }

    @Test
    void findRescheduleTest() {

        Collection<PrimaryId> primaryIds =
                storage.find().whereNextRunLeNow()
                        .whereNeedReschedule(false)
                        .whereJobStatus(JobStatus.READY)
                        .findPrimaryIds();

        Collection<Job> jobs = storage.find()
                .wherePrimaryIdIn(primaryIds)
                .whereJobStatus(JobStatus.READY).findJobs();

        assertThat(jobs).hasSize(2);
        var expectedIds = List.of(1L, 2L);
        var gotIds = jobs.stream().map(job -> ((PrimaryIdImpl) job.primaryId()).getId()).collect(toList());
        assertThat(gotIds).containsExactlyInAnyOrder(expectedIds.toArray(Long[]::new));
    }

    @Test
    void updateTest() {
        storage.update().whereNextRunLeNow().whereJobStatus(JobStatus.READY).setJobStatus(JobStatus.LOCKED)
                .execute();

        Map<Long, LocalDateTime> result =
                dslContext.select(SCHEDULED_TASKS.ID, SCHEDULED_TASKS.HEARTBEAT_TIME)
                        .from(SCHEDULED_TASKS)
                        .where(SCHEDULED_TASKS.STATUS.eq(Running))
                        .and(SCHEDULED_TASKS.INSTANCE_ID.eq(schedulerId.toString()))
                        .fetchMap(r -> r.get(SCHEDULED_TASKS.ID), r -> r.get(SCHEDULED_TASKS.HEARTBEAT_TIME));

        assertThat(result).hasSize(4);
        var expectedUpdatedIds = List.of(1L, 2L, 3L);
        var expectedNotUpdatedIds = List.of(8L);
        assertThat(result).containsKeys(expectedUpdatedIds.toArray(Long[]::new));
        assertThat(result).containsKeys(expectedNotUpdatedIds.toArray(Long[]::new));

        var offset = new TemporalUnitWithinOffset(30, ChronoUnit.SECONDS);
        for (var expectedUpdatedId : expectedUpdatedIds) {
            assertThat(result.get(expectedUpdatedId)).isCloseTo(LocalDateTime.now(), offset);
        }

        for (var expectedNotUpdatedId : expectedNotUpdatedIds) {
            assertThat(result.get(expectedNotUpdatedId)).isBefore(NOW.minusMinutes(1));
        }
    }
}
