package ru.yandex.direct.hourglass.mysql.storage;

import java.sql.SQLException;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import ru.yandex.direct.hourglass.HourglassProperties;
import ru.yandex.direct.hourglass.implementations.InstanceIdImpl;
import ru.yandex.direct.hourglass.implementations.updateschedule.ScheduleRecord;
import ru.yandex.direct.hourglass.mysql.DslContextHolder;
import ru.yandex.partner.dbschema.partner.tables.records.ScheduledTasksRecord;

import static org.assertj.core.api.Assertions.assertThat;
import static ru.yandex.partner.dbschema.partner.Tables.SCHEDULED_TASKS;
import static ru.yandex.partner.dbschema.partner.enums.ScheduledTasksStatus.Deleted;
import static ru.yandex.partner.dbschema.partner.enums.ScheduledTasksStatus.New;
import static ru.yandex.partner.dbschema.partner.enums.ScheduledTasksStatus.Paused;
import static ru.yandex.partner.dbschema.partner.enums.ScheduledTasksStatus.Running;

class ScheduleUpdateTest {

    private static final String OLD_VERSION = "old_version";
    private static final String NEW_VERSION = "new_version";
    private static final List<Field> SCHEDULED_TASK_FIELDS = List.of(SCHEDULED_TASKS.ID,
            SCHEDULED_TASKS.NAME,
            SCHEDULED_TASKS.PARAMS,
            SCHEDULED_TASKS.STATUS,
            SCHEDULED_TASKS.INSTANCE_ID,
            SCHEDULED_TASKS.HEARTBEAT_TIME,
            SCHEDULED_TASKS.SCHEDULE_HASH,
            SCHEDULED_TASKS.JOB_NAME_HASH,
            SCHEDULED_TASKS.NEED_RESCHEDULE,
            SCHEDULED_TASKS.NEXT_RUN,
            SCHEDULED_TASKS.LAST_START_TIME,
            SCHEDULED_TASKS.LAST_FINISH_TIME,
            SCHEDULED_TASKS.VERSION,
            SCHEDULED_TASKS.META);
    private static StorageImpl storage;
    private static DSLContext dslContext;

    @BeforeAll
    static void initDb() throws SQLException, InterruptedException {
        dslContext = DslContextHolder.getDslContext();
        var instanceId = new InstanceIdImpl("instance");

        storage = new StorageImpl(dslContext, instanceId, HourglassProperties.builder().build(), NEW_VERSION);
    }

    /*
     * Мы будем проверять что произойдет с заданиями в разных состояниях при изменении  расписания
     */
    @BeforeEach
    void makeSchedule() {
        dslContext.truncate(SCHEDULED_TASKS).execute();
    }

    /**
     * Тест проверяет, что расписние менсяется только у несовпадающий версий
     */
    @Test
    void setNewSchedule_OnlyDifferentVersionUpdateTest() {

        var recordWithSameVersion = new ScheduledTasksRecord();
        recordWithSameVersion.setId(1L);
        recordWithSameVersion.setName("name1");
        recordWithSameVersion.setParams("param");
        recordWithSameVersion.setStatus(New);
        recordWithSameVersion.setNeedReschedule(0L);
        recordWithSameVersion.setHeartbeatTime(LocalDateTime.of(2019, 9, 30, 1, 2));
        recordWithSameVersion.setNextRun(LocalDateTime.of(2019, 9, 30, 1, 2, 30));
        recordWithSameVersion.setScheduleHash("hash");
        recordWithSameVersion.setJobNameHash("nameHash");
        recordWithSameVersion.setVersion(NEW_VERSION);
        recordWithSameVersion.setLastFinishTime(LocalDateTime.of(2019, 9, 30, 0, 0, 30));
        recordWithSameVersion.setLastStartTime(LocalDateTime.of(2019, 9, 30, 0, 0, 20));
        recordWithSameVersion.setMeta("meta1_old");

        var recordWithAnotherVersion = new ScheduledTasksRecord();
        recordWithAnotherVersion.setId(2L);
        recordWithAnotherVersion.setName("name2");
        recordWithAnotherVersion.setParams("");
        recordWithAnotherVersion.setStatus(New);
        recordWithAnotherVersion.setNeedReschedule(0L);
        recordWithAnotherVersion.setHeartbeatTime(LocalDateTime.of(2019, 9, 30, 0, 59, 21));
        recordWithAnotherVersion.setNextRun(LocalDateTime.of(2019, 9, 30, 4, 2, 36));
        recordWithAnotherVersion.setScheduleHash("hash_old");
        recordWithAnotherVersion.setJobNameHash("nameHash");
        recordWithAnotherVersion.setVersion(OLD_VERSION);
        recordWithAnotherVersion.setLastFinishTime(LocalDateTime.of(2019, 9, 30, 3, 29, 30));
        recordWithAnotherVersion.setLastStartTime(LocalDateTime.of(2019, 9, 30, 3, 13, 45));
        recordWithAnotherVersion.setMeta("meta2_old");

        insertSchedule(recordWithSameVersion, recordWithAnotherVersion);

        var newSchedule = List.of(
                new ScheduleRecord().setName("name1").setParam("param").setScheduleHashSum("hash").setNameHashSum(
                        "nameHash").setMeta("meta1_old"),
                new ScheduleRecord().setName("name2").setParam("").setScheduleHashSum("hash_new").setNameHashSum(
                        "nameHash").setMeta("meta2_new")
        );

        storage.setNewSchedule(newSchedule);

        var idToScheduleRecord = getIdScheduleMap();

        assertThat(idToScheduleRecord).containsKeys(1L, 2L);
        assertThat(idToScheduleRecord.get(1L)).isEqualTo(recordWithSameVersion);
        var differentScheduleRecord = idToScheduleRecord.get(2L);
        assertRecordsEquals(differentScheduleRecord, recordWithAnotherVersion,
                Set.of(SCHEDULED_TASKS.NEED_RESCHEDULE, SCHEDULED_TASKS.VERSION, SCHEDULED_TASKS.SCHEDULE_HASH,
                        SCHEDULED_TASKS.META));
        assertThat(differentScheduleRecord.getNeedReschedule()).isEqualTo(1L);
        assertThat(differentScheduleRecord.getVersion()).isEqualTo(NEW_VERSION);
        assertThat(differentScheduleRecord.getScheduleHash()).isEqualTo("hash_new");
        assertThat(differentScheduleRecord.getMeta()).isEqualTo("meta2_new");
    }

    /**
     * Тест проверяет, что расписние менсяется только у задачи, у которых новое расписание отличается от старого, но
     * версия обновится у всех∆
     */
    @Test
    void setNewSchedule_OnlyWhenDifferentScheduleSumUpdateTest() {

        var recordWithSameVersion = new ScheduledTasksRecord();
        recordWithSameVersion.setId(1L);
        recordWithSameVersion.setName("name1");
        recordWithSameVersion.setParams("param");
        recordWithSameVersion.setStatus(New);
        recordWithSameVersion.setNeedReschedule(0L);
        recordWithSameVersion.setHeartbeatTime(LocalDateTime.of(2019, 9, 30, 1, 2));
        recordWithSameVersion.setNextRun(LocalDateTime.of(2019, 9, 30, 1, 2, 30));
        recordWithSameVersion.setScheduleHash("hash");
        recordWithSameVersion.setJobNameHash("nameHash");
        recordWithSameVersion.setVersion(OLD_VERSION);
        recordWithSameVersion.setLastFinishTime(LocalDateTime.of(2019, 9, 30, 1, 0, 30));
        recordWithSameVersion.setLastStartTime(LocalDateTime.of(2019, 9, 30, 0, 0, 30));


        var recordWithAnotherVersion = new ScheduledTasksRecord();
        recordWithAnotherVersion.setId(2L);
        recordWithAnotherVersion.setName("name2");
        recordWithAnotherVersion.setParams("");
        recordWithAnotherVersion.setStatus(New);
        recordWithAnotherVersion.setNeedReschedule(0L);
        recordWithAnotherVersion.setHeartbeatTime(LocalDateTime.of(2019, 9, 30, 0, 59, 21));
        recordWithAnotherVersion.setNextRun(LocalDateTime.of(2019, 9, 30, 4, 2, 36));
        recordWithAnotherVersion.setScheduleHash("hash_old");
        recordWithAnotherVersion.setJobNameHash("nameHash");
        recordWithAnotherVersion.setVersion(OLD_VERSION);
        recordWithAnotherVersion.setLastFinishTime(LocalDateTime.of(2019, 9, 30, 2, 17, 11));
        recordWithAnotherVersion.setLastStartTime(LocalDateTime.of(2019, 9, 30, 2, 15, 1));


        insertSchedule(recordWithSameVersion, recordWithAnotherVersion);

        var newSchedule = List.of(
                new ScheduleRecord().setName("name1").setParam("param").setScheduleHashSum("hash").setNameHashSum(
                        "nameHash"),
                new ScheduleRecord().setName("name2").setParam("").setScheduleHashSum("hash_new").setNameHashSum(
                        "nameHash")
        );

        storage.setNewSchedule(newSchedule);

        var idToScheduleRecord = getIdScheduleMap();

        assertThat(idToScheduleRecord).hasSize(2);
        assertThat(idToScheduleRecord).containsKeys(1L, 2L);
        var sameScheduleRecord = idToScheduleRecord.get(1L);
        assertRecordsEquals(sameScheduleRecord, recordWithSameVersion, Set.of(SCHEDULED_TASKS.VERSION));
        assertThat(sameScheduleRecord.getVersion()).isEqualTo(NEW_VERSION);

        var differentScheduleRecord = idToScheduleRecord.get(2L);
        assertRecordsEquals(differentScheduleRecord, recordWithAnotherVersion,
                Set.of(SCHEDULED_TASKS.NEED_RESCHEDULE, SCHEDULED_TASKS.VERSION, SCHEDULED_TASKS.SCHEDULE_HASH));
        assertThat(differentScheduleRecord.getNeedReschedule()).isEqualTo(1L);
        assertThat(differentScheduleRecord.getVersion()).isEqualTo(NEW_VERSION);
        assertThat(differentScheduleRecord.getScheduleHash()).isEqualTo("hash_new");
    }

    /**
     * Тест проверяет, что если расписание задачи не поменялось, а поле meta поменялось - оно обновится
     */
    @Test
    void setNewSchedule_MetaUpdateTest() {

        var record = new ScheduledTasksRecord();
        record.setId(1L);
        record.setName("name1");
        record.setParams("param");
        record.setStatus(New);
        record.setNeedReschedule(0L);
        record.setHeartbeatTime(LocalDateTime.of(2019, 9, 30, 1, 2));
        record.setNextRun(LocalDateTime.of(2019, 9, 30, 1, 2, 30));
        record.setScheduleHash("hash");
        record.setJobNameHash("nameHash");
        record.setVersion(OLD_VERSION);
        record.setLastFinishTime(LocalDateTime.of(2019, 9, 30, 1, 0, 30));
        record.setLastStartTime(LocalDateTime.of(2019, 9, 30, 0, 0, 30));
        record.setMeta("meta_old");

        insertSchedule(record);

        var newSchedule = List.of(
                new ScheduleRecord().setName("name1").setParam("param").setScheduleHashSum("hash").setNameHashSum(
                        "nameHash").setMeta("meta_new")
        );

        storage.setNewSchedule(newSchedule);

        var idToScheduleRecord = getIdScheduleMap();

        assertThat(idToScheduleRecord).hasSize(1);
        assertThat(idToScheduleRecord).containsKeys(1L);
        var gotRecord = idToScheduleRecord.get(1L);
        assertRecordsEquals(gotRecord, record, Set.of(SCHEDULED_TASKS.VERSION, SCHEDULED_TASKS.META));
        assertThat(gotRecord.getVersion()).isEqualTo(NEW_VERSION);
        assertThat(gotRecord.getMeta()).isEqualTo("meta_new");
    }

    /**
     * Тест проверяет, вставку новой задачи
     */
    @Test
    void setNewSchedule_InsertNewTask() {

        var newSchedule = List.of(
                new ScheduleRecord().setName("name1").setParam("param").setScheduleHashSum("hash").setNameHashSum(
                        "nameHash")
        );

        storage.setNewSchedule(newSchedule);

        var idToScheduleRecord = getIdScheduleMap();

        assertThat(idToScheduleRecord).hasSize(1);
        var id = idToScheduleRecord.keySet().iterator().next();
        var gotRecord = idToScheduleRecord.get(id);
        var expectedRecord = new ScheduledTasksRecord();
        expectedRecord.setId(id);
        expectedRecord.setName("name1");
        expectedRecord.setParams("param");
        expectedRecord.setScheduleHash("hash");
        expectedRecord.setJobNameHash("nameHash");
        expectedRecord.setNeedReschedule(1L);
        expectedRecord.setVersion(NEW_VERSION);
        expectedRecord.setStatus(New);

        assertThat(gotRecord).isEqualTo(expectedRecord);
    }

    /**
     * Тест проверяет, что статусы Running и Paused не изменятся при обновлении расписания
     */
    @Test
    void setNewSchedule_StatusNotChangedTest() {

        var taskWithStatusRunningBeforeUpdate = new ScheduledTasksRecord();
        taskWithStatusRunningBeforeUpdate.setId(1L);
        taskWithStatusRunningBeforeUpdate.setName("name1");
        taskWithStatusRunningBeforeUpdate.setParams("param");
        taskWithStatusRunningBeforeUpdate.setStatus(Running);
        taskWithStatusRunningBeforeUpdate.setNeedReschedule(0L);
        taskWithStatusRunningBeforeUpdate.setHeartbeatTime(LocalDateTime.of(2019, 9, 30, 1, 2));
        taskWithStatusRunningBeforeUpdate.setNextRun(LocalDateTime.of(2019, 9, 30, 1, 2, 30));
        taskWithStatusRunningBeforeUpdate.setScheduleHash("hash_old1");
        taskWithStatusRunningBeforeUpdate.setInstanceId("some_instance");
        taskWithStatusRunningBeforeUpdate.setJobNameHash("nameHash");
        taskWithStatusRunningBeforeUpdate.setVersion(OLD_VERSION);
        taskWithStatusRunningBeforeUpdate.setLastFinishTime(LocalDateTime.of(2019, 9, 30, 1, 0, 30));
        taskWithStatusRunningBeforeUpdate.setLastStartTime(LocalDateTime.of(2019, 9, 30, 0, 0, 30));


        var taskWithStatusPausedBeforeUpdate = new ScheduledTasksRecord();
        taskWithStatusPausedBeforeUpdate.setId(2L);
        taskWithStatusPausedBeforeUpdate.setName("name2");
        taskWithStatusPausedBeforeUpdate.setParams("");
        taskWithStatusPausedBeforeUpdate.setStatus(Paused);
        taskWithStatusPausedBeforeUpdate.setNeedReschedule(0L);
        taskWithStatusPausedBeforeUpdate.setHeartbeatTime(LocalDateTime.of(2019, 9, 30, 0, 59, 21));
        taskWithStatusPausedBeforeUpdate.setNextRun(LocalDateTime.of(2019, 9, 30, 4, 2, 36));
        taskWithStatusPausedBeforeUpdate.setScheduleHash("hash_old2");
        taskWithStatusPausedBeforeUpdate.setJobNameHash("nameHash");
        taskWithStatusPausedBeforeUpdate.setVersion(OLD_VERSION);
        taskWithStatusPausedBeforeUpdate.setLastFinishTime(LocalDateTime.of(2019, 9, 30, 1, 0, 30));
        taskWithStatusPausedBeforeUpdate.setLastStartTime(LocalDateTime.of(2019, 9, 30, 0, 0, 30));


        insertSchedule(taskWithStatusRunningBeforeUpdate, taskWithStatusPausedBeforeUpdate);

        var newSchedule = List.of(
                new ScheduleRecord().setName("name1").setParam("param").setScheduleHashSum("hash_new1").setNameHashSum(
                        "nameHash"),
                new ScheduleRecord().setName("name2").setParam("").setScheduleHashSum("hash_new2").setNameHashSum(
                        "nameHash")
        );

        storage.setNewSchedule(newSchedule);

        var idToScheduleRecord = getIdScheduleMap();

        assertThat(idToScheduleRecord).containsKeys(1L, 2L);
        var taskWithStatusRunningAfterUpdate = idToScheduleRecord.get(1L);
        assertRecordsEquals(taskWithStatusRunningAfterUpdate, taskWithStatusRunningBeforeUpdate,
                Set.of(SCHEDULED_TASKS.NEED_RESCHEDULE, SCHEDULED_TASKS.VERSION, SCHEDULED_TASKS.SCHEDULE_HASH));
        assertThat(taskWithStatusRunningAfterUpdate.getVersion()).isEqualTo(NEW_VERSION);
        assertThat(taskWithStatusRunningAfterUpdate.getScheduleHash()).isEqualTo("hash_new1");
        assertThat(taskWithStatusRunningAfterUpdate.getNeedReschedule()).isEqualTo(1L);

        var taskWithStatusPausedAfterUpdate = idToScheduleRecord.get(2L);
        assertRecordsEquals(taskWithStatusPausedAfterUpdate, taskWithStatusPausedBeforeUpdate,
                Set.of(SCHEDULED_TASKS.NEED_RESCHEDULE, SCHEDULED_TASKS.VERSION, SCHEDULED_TASKS.SCHEDULE_HASH));
        assertThat(taskWithStatusPausedAfterUpdate.getNeedReschedule()).isEqualTo(1L);
        assertThat(taskWithStatusPausedAfterUpdate.getVersion()).isEqualTo(NEW_VERSION);
        assertThat(taskWithStatusPausedAfterUpdate.getScheduleHash()).isEqualTo("hash_new2");
    }

    /**
     * Тест проверяет, что задача есть в базе, но в новом расписании ее нет - она заархивируется, но версия у нее
     * обновится
     */
    @Test
    void setNewSchedule_ArchiveTaskTest() {

        var taskToBeArchived = new ScheduledTasksRecord();
        taskToBeArchived.setId(1L);
        taskToBeArchived.setName("name1");
        taskToBeArchived.setParams("param");
        taskToBeArchived.setStatus(New);
        taskToBeArchived.setNeedReschedule(0L);
        taskToBeArchived.setHeartbeatTime(LocalDateTime.of(2019, 9, 30, 1, 2));
        taskToBeArchived.setNextRun(LocalDateTime.of(2019, 9, 30, 1, 2, 30));
        taskToBeArchived.setScheduleHash("hash_old1");
        taskToBeArchived.setJobNameHash("nameHash");
        taskToBeArchived.setVersion(OLD_VERSION);
        taskToBeArchived.setLastFinishTime(LocalDateTime.of(2019, 9, 30, 1, 0, 30));
        taskToBeArchived.setLastStartTime(LocalDateTime.of(2019, 9, 30, 0, 0, 30));


        insertSchedule(taskToBeArchived);

        var newSchedule = List.of(
                new ScheduleRecord().setName("name2").setParam("").setScheduleHashSum("hash_new2").setNameHashSum(
                        "nameHash")
        );

        storage.setNewSchedule(newSchedule);

        var idToScheduleRecord = getIdScheduleMap();

        assertThat(idToScheduleRecord).containsKey(1L);
        var taskWithStatusRunningAfterUpdate = idToScheduleRecord.get(1L);
        assertRecordsEquals(taskWithStatusRunningAfterUpdate, taskToBeArchived, Set.of(SCHEDULED_TASKS.STATUS,
                SCHEDULED_TASKS.VERSION));
        assertThat(taskWithStatusRunningAfterUpdate.getVersion()).isEqualTo(NEW_VERSION);
        assertThat(taskWithStatusRunningAfterUpdate.getStatus()).isEqualTo(Deleted);
    }

    /**
     * Тест проверяет, что при обновлении расписания статус needReschedule не сброится на false
     */
    @Test
    void setNewSchedule_NeedRescheduleNotResetTest() {

        var needRescheduleTask = new ScheduledTasksRecord();
        needRescheduleTask.setId(1L);
        needRescheduleTask.setName("name1");
        needRescheduleTask.setParams("param");
        needRescheduleTask.setStatus(New);
        needRescheduleTask.setNeedReschedule(0L);
        needRescheduleTask.setHeartbeatTime(LocalDateTime.of(2019, 9, 30, 1, 2));
        needRescheduleTask.setNextRun(LocalDateTime.of(2019, 9, 30, 1, 2, 30));
        needRescheduleTask.setScheduleHash("hash_old1");
        needRescheduleTask.setJobNameHash("nameHash");
        needRescheduleTask.setNeedReschedule(1L);
        needRescheduleTask.setVersion(OLD_VERSION);
        needRescheduleTask.setLastFinishTime(LocalDateTime.of(2019, 9, 30, 1, 0, 30));
        needRescheduleTask.setLastStartTime(LocalDateTime.of(2019, 9, 30, 0, 0, 30));

        var needRescheduleButNotChangedTask = new ScheduledTasksRecord();
        needRescheduleButNotChangedTask.setId(2L);
        needRescheduleButNotChangedTask.setName("name2");
        needRescheduleButNotChangedTask.setParams("");
        needRescheduleButNotChangedTask.setStatus(New);
        needRescheduleButNotChangedTask.setNeedReschedule(1L);
        needRescheduleButNotChangedTask.setHeartbeatTime(LocalDateTime.of(2019, 9, 30, 1, 2));
        needRescheduleButNotChangedTask.setNextRun(LocalDateTime.of(2019, 9, 30, 1, 2, 30));
        needRescheduleButNotChangedTask.setScheduleHash("hash");
        needRescheduleButNotChangedTask.setJobNameHash("nameHash2");
        needRescheduleButNotChangedTask.setNeedReschedule(1L);
        needRescheduleButNotChangedTask.setVersion(OLD_VERSION);
        needRescheduleButNotChangedTask.setLastFinishTime(LocalDateTime.of(2019, 9, 30, 1, 0, 30));
        needRescheduleButNotChangedTask.setLastStartTime(LocalDateTime.of(2019, 9, 30, 0, 0, 30));

        insertSchedule(needRescheduleTask, needRescheduleButNotChangedTask);

        var newSchedule = List.of(
                new ScheduleRecord().setName("name1").setParam("param").setScheduleHashSum("hash_new2").setNameHashSum(
                        "nameHash"),
                new ScheduleRecord().setName("name2").setParam("").setScheduleHashSum("hash").setNameHashSum(
                        "nameHash2")
        );

        storage.setNewSchedule(newSchedule);

        var idToScheduleRecord = getIdScheduleMap();

        assertThat(idToScheduleRecord).hasSize(2);
        assertThat(idToScheduleRecord).containsOnlyKeys(1L, 2L);
        var needRescheduleTaskGot = idToScheduleRecord.get(1L);
        assertRecordsEquals(needRescheduleTaskGot, needRescheduleTask, Set.of(SCHEDULED_TASKS.SCHEDULE_HASH,
                SCHEDULED_TASKS.VERSION));
        assertThat(needRescheduleTaskGot.getScheduleHash()).isEqualTo("hash_new2");
        assertThat(needRescheduleTaskGot.getVersion()).isEqualTo(NEW_VERSION);

        /* needReschedule не должет сброситься */
        var needRescheduleButNotChangedTaskGot = idToScheduleRecord.get(2L);
        assertRecordsEquals(needRescheduleButNotChangedTaskGot, needRescheduleButNotChangedTaskGot,
                Set.of(SCHEDULED_TASKS.VERSION));
        assertThat(needRescheduleButNotChangedTaskGot.getVersion()).isEqualTo(NEW_VERSION);
    }

    @Test
    void setNewSchedule_UnArchiveTaskTest() {

        var taskToBeUnarchived = new ScheduledTasksRecord();
        taskToBeUnarchived.setId(1L);
        taskToBeUnarchived.setName("name1");
        taskToBeUnarchived.setParams("param");
        taskToBeUnarchived.setStatus(Deleted);
        taskToBeUnarchived.setNeedReschedule(0L);
        taskToBeUnarchived.setHeartbeatTime(LocalDateTime.of(2019, 9, 30, 1, 2));
        taskToBeUnarchived.setNextRun(LocalDateTime.of(2019, 9, 30, 1, 2, 30));
        taskToBeUnarchived.setScheduleHash("hash_old1");
        taskToBeUnarchived.setJobNameHash("nameHash");
        taskToBeUnarchived.setVersion(OLD_VERSION);
        taskToBeUnarchived.setInstanceId("instance_id");
        taskToBeUnarchived.setLastFinishTime(LocalDateTime.of(2019, 9, 30, 1, 0, 30));
        taskToBeUnarchived.setLastStartTime(LocalDateTime.of(2019, 9, 30, 0, 0, 30));


        insertSchedule(taskToBeUnarchived);

        var newSchedule = List.of(
                new ScheduleRecord().setName("name1").setParam("param").setScheduleHashSum("hash_new").setNameHashSum(
                        "nameHash")
        );

        storage.setNewSchedule(newSchedule);

        var idToScheduleRecord = getIdScheduleMap();

        assertThat(idToScheduleRecord).containsKey(1L);
        var taskWithStatusRunningAfterUpdate = idToScheduleRecord.get(1L);
        assertRecordsEquals(taskWithStatusRunningAfterUpdate, taskToBeUnarchived, Set.of(SCHEDULED_TASKS.STATUS,
                SCHEDULED_TASKS.VERSION, SCHEDULED_TASKS.SCHEDULE_HASH, SCHEDULED_TASKS.NEED_RESCHEDULE,
                SCHEDULED_TASKS.INSTANCE_ID));
        assertThat(taskWithStatusRunningAfterUpdate.getVersion()).isEqualTo(NEW_VERSION);
        assertThat(taskWithStatusRunningAfterUpdate.getStatus()).isEqualTo(New);
        assertThat(taskWithStatusRunningAfterUpdate.getScheduleHash()).isEqualTo("hash_new");
        assertThat(taskWithStatusRunningAfterUpdate.getNeedReschedule()).isEqualTo(1L);
        assertThat(taskWithStatusRunningAfterUpdate.getInstanceId()).isNull();
    }


    private void assertRecordsEquals(ScheduledTasksRecord actual,
                                     ScheduledTasksRecord expected, Set<Field> fieldsNotEquals) {
        for (var field : SCHEDULED_TASKS.fields()) {
            if (!fieldsNotEquals.contains(field)) {
                assertThat(actual.get(field)).isEqualTo(expected.get(field));
            }
        }
    }

    private void insertSchedule(ScheduledTasksRecord... scheduledTasksRecords) {
        dslContext.batchInsert(scheduledTasksRecords)
                .execute();
    }

    private Map<Long, ScheduledTasksRecord> getIdScheduleMap() {
        return dslContext
                .select(SCHEDULED_TASK_FIELDS)
                .from(SCHEDULED_TASKS)
                .fetchMap(r -> r.get(SCHEDULED_TASKS.ID),
                        r -> {
                            var scheduledTasksRecord = new ScheduledTasksRecord();
                            for (Field field : SCHEDULED_TASK_FIELDS) {
                                var value = r.get(field);
                                scheduledTasksRecord.with(field, value);
                            }
                            return scheduledTasksRecord;
                        });
    }
}
