package ru.yandex.partner.libs.i18n;

import java.util.Locale;
import java.util.ResourceBundle;

import javax.annotation.Nonnull;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.MessageSourceResolvable;
import org.springframework.context.support.DelegatingMessageSource;

import ru.yandex.partner.libs.i18n.gettext.Gettext;
import ru.yandex.partner.libs.i18n.gettext.GettextImpl;
import ru.yandex.partner.libs.i18n.resolver.bundle.DashResourceBundleNameResolver;
import ru.yandex.partner.libs.i18n.resolver.bundle.ResourceBundleNameResolver;
import ru.yandex.partner.libs.i18n.tanker.bundle.TankerJsonResourceBundleControl;
import ru.yandex.partner.libs.i18n.tanker.escaper.TankerPOStringEscaper;
import ru.yandex.partner.libs.i18n.tanker.resolver.TankerGettextMessageKeyResolver;

public class GettextMessageSource extends DelegatingMessageSource {

    private static final Logger LOGGER = LoggerFactory.getLogger(GettextMessageSource.class);

    private final Gettext gettext;
    private final ResourceBundleNameResolver resourceBundleNameResolver;
    private final TankerJsonResourceBundleControl resourceBundleControl;

    public GettextMessageSource(String baseResourcePath) {
        this(new GettextImpl(new TankerGettextMessageKeyResolver()),
                new DashResourceBundleNameResolver(baseResourcePath),
                new TankerJsonResourceBundleControl(new TankerPOStringEscaper(), true));
    }

    public GettextMessageSource(Gettext gettext, ResourceBundleNameResolver resourceBundleNameResolver,
                                TankerJsonResourceBundleControl resourceBundleControl) {
        this.gettext = gettext;
        this.resourceBundleNameResolver = resourceBundleNameResolver;
        this.resourceBundleControl = resourceBundleControl;
    }

    @Nonnull
    @Override
    public String getMessage(@Nonnull String code, Object[] args, @Nonnull Locale locale) {
        return super.getMessage(code, args, locale);
    }

    @Override
    public String getMessage(@Nonnull String code, Object[] args, String defaultMessage, @Nonnull Locale locale) {
        return super.getMessage(code, args, defaultMessage, locale);
    }

    @Nonnull
    @Override
    public String getMessage(@Nonnull MessageSourceResolvable resolvable, @Nonnull Locale locale) {
        if (resolvable instanceof GettextMsg) {
            String text = getText((GettextMsg) resolvable, locale);
            return String.format(text, resolvable.getArguments());
        } else {
            return super.getMessage(resolvable, locale);
        }
    }

    @Nonnull
    private String getText(GettextMsg gettextMsg, Locale locale) {

        final GettextMsgPayload payload = gettextMsg.getPayload();

        final String bundleName = resourceBundleNameResolver.resolveResourceBundleName(gettextMsg.getKeysetName());
        final ResourceBundle resourceBundle = ResourceBundle.getBundle(bundleName, locale, resourceBundleControl);

        int n = 0;
        if (payload.isPlural()) {
            int argIndex = payload.getPluralArgIndex();
            Object[] arguments = gettextMsg.getArguments();
            if (arguments == null || argIndex < 0 || argIndex >= arguments.length || !(arguments[argIndex] instanceof Number)) {
                LOGGER.warn("Expected numeric argument for plural form on index {}. Message: {}, arguments: {}",
                        argIndex, payload.getMsg(), arguments);
                n = 0;
            } else {
                n = ((Number) arguments[argIndex]).intValue();
            }
        }

        if (payload.isPlural() && payload.isContext()) {
            return gettext.npGetText(resourceBundle, payload.getMsg(), n, payload.getPluralMsg(), payload.getContext());
        } else if (payload.isPlural()) {
            return gettext.nGetText(resourceBundle, payload.getMsg(), n, payload.getPluralMsg());
        } else if (payload.isContext()) {
            return gettext.pGetText(resourceBundle, payload.getMsg(), payload.getContext());
        } else {
            return gettext.getText(resourceBundle, payload.getMsg());
        }
    }

}
