package ru.yandex.partner.libs.i18n.tanker.bundle;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.ResourceBundle;

import javax.annotation.Nonnull;

import org.springframework.context.i18n.LocaleContextHolder;

import ru.yandex.partner.libs.i18n.escaper.StringEscaper;
import ru.yandex.partner.libs.i18n.tanker.exception.TankerJsonException;

/**
 * Inspired by example at https://docs.oracle.com/javase/7/docs/api/java/util/ResourceBundle.Control.html
 */
public class TankerJsonResourceBundleControl extends ResourceBundle.Control {

    public static final String FORMAT_JSON = "json";

    private final StringEscaper stringEscaper;
    private final boolean fallbackToDefaultLocale;

    public TankerJsonResourceBundleControl(StringEscaper stringEscaper, boolean fallbackToDefaultLocale) {
        this.stringEscaper = stringEscaper;
        this.fallbackToDefaultLocale = fallbackToDefaultLocale;
    }

    @Override
    public List<String> getFormats(@Nonnull String baseName) {
        Objects.requireNonNull(baseName);
        return List.of(FORMAT_JSON);
    }

    @Override
    public Locale getFallbackLocale(@Nonnull String baseName, @Nonnull Locale locale) {
        Objects.requireNonNull(baseName);
        Objects.requireNonNull(locale);
        if (!fallbackToDefaultLocale) {
            return null;
        }
        Locale defaultLocale = LocaleContextHolder.getLocale();
        return locale.equals(defaultLocale) ? null : defaultLocale;
    }

    public ResourceBundle newBundle(@Nonnull String baseName,
                                    @Nonnull Locale locale,
                                    @Nonnull String format,
                                    @Nonnull ClassLoader loader,
                                    boolean reload)
            throws IOException {
        Objects.requireNonNull(baseName);
        Objects.requireNonNull(locale);
        Objects.requireNonNull(format);
        Objects.requireNonNull(loader);

        ResourceBundle bundle = null;
        if (format.equals(FORMAT_JSON)) {
            String bundleName = toBundleName(baseName, locale);
            String resourceName = toResourceName(bundleName, format);
            InputStream stream = null;
            if (reload) {
                URL url = loader.getResource(resourceName);
                if (url != null) {
                    URLConnection connection = url.openConnection();
                    if (connection != null) {
                        // Disable caches to get fresh data for
                        // reloading.
                        connection.setUseCaches(false);
                        stream = connection.getInputStream();
                    }
                }
            } else {
                stream = loader.getResourceAsStream(resourceName);
            }
            if (stream != null) {
                try (BufferedInputStream bis = new BufferedInputStream(stream)) {
                    bundle = new TankerJsonResourceBundle(bis, stringEscaper);
                } catch (TankerJsonException e) {
                    throw new TankerJsonException(String.format("Error while loading resource %s", resourceName), e);
                }
            }
        }
        return bundle;
    }
}
