package ru.yandex.partner.libs.i18n.tanker.client;

import java.time.Duration;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.client.MultipartBodyBuilder;
import org.springframework.web.reactive.function.BodyInserters;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.reactive.function.client.WebClientResponseException;
import org.springframework.web.util.UriComponentsBuilder;
import reactor.core.publisher.Mono;
import reactor.util.retry.Retry;

import ru.yandex.partner.libs.i18n.tanker.dto.json.TankerJsonKeysetDoc;
import ru.yandex.partner.libs.i18n.tanker.dto.tjson.TankerDetailedJsonDoc;

import static org.springframework.http.HttpStatus.TOO_MANY_REQUESTS;

public class TankerClientImpl implements TankerClient {

    private static final Logger LOGGER = LoggerFactory.getLogger(TankerClientImpl.class);

    private static final String DOWNLOAD_KEYSET_URL = "keysets/json/";
    private static final String UPLOAD_URL = "keysets/merge/";

    private final WebClient webClient;
    private final String baseTankerUrl;
    private final String projectId;
    private final String branchId;
    private final String accessToken;

    public TankerClientImpl(WebClient webClient, String baseTankerUrl, String projectId, String branchId,
                            String accessToken) {
        this.webClient = webClient;
        this.baseTankerUrl = baseTankerUrl;
        this.branchId = branchId;
        this.projectId = projectId;
        this.accessToken = accessToken;
        LOGGER.debug("Tanker client works with project '{}', branch '{}'", projectId, branchId);
    }

    public TankerClientImpl(WebClient webClient, String baseTankerUrl, String projectId, String branchId) {
        this(webClient, baseTankerUrl, projectId, branchId, null);
    }

    @Override
    public TankerJsonKeysetDoc downloadKeyset(String keysetId, String language) throws TankerClientException {
        UriComponentsBuilder uriBuilder = UriComponentsBuilder
                .fromUriString(baseTankerUrl)
                .path(DOWNLOAD_KEYSET_URL)
                .queryParam("project-id", projectId)
                .queryParam("branch-id", branchId)
                .queryParam("flat-keyset", 1)
                .queryParam("status", "unapproved")
                .queryParam("keyset-id", keysetId)
                .queryParam("language", language);

        return webClient.get()
                .uri(uriBuilder.build().toUri())
                .accept(MediaType.APPLICATION_JSON)
                .retrieve()
                .onStatus(HttpStatus.NOT_FOUND::equals, response -> {
                    LOGGER.warn("Could not download translations for keyset {}: {}", keysetId, HttpStatus.NOT_FOUND);
                    return Mono.empty();
                })
                .toEntity(TankerJsonKeysetDoc.class)
                .retryWhen(Retry.backoff(3, Duration.ofSeconds(1))
                        .filter(e -> ((WebClientResponseException) e)
                                .getStatusCode().equals(TOO_MANY_REQUESTS))
                )
                .onErrorMap(e -> {
                    LOGGER.error("Could not download translations for keyset {}: {}", keysetId, e.getMessage());
                    return new TankerClientException(e);
                })
                .blockOptional()
                .map(HttpEntity::getBody)
                .orElse(null);
    }

    @Override
    public void uploadTranslations(TankerDetailedJsonDoc tankerDetailedJsonDoc) throws TankerClientException {
        try {
            UriComponentsBuilder uriBuilder = UriComponentsBuilder
                    .fromUriString(baseTankerUrl)
                    .path(UPLOAD_URL);

            String tJsonValue = encodeDocument(tankerDetailedJsonDoc);
            LOGGER.debug("Content of uploaded file: {}", tJsonValue);

            MultipartBodyBuilder multipart = new MultipartBodyBuilder();
            multipart.part("project-id", projectId);
            multipart.part("branch-id", branchId);
            multipart.part("format", "tjson");
            multipart.part("file", tJsonValue, MediaType.APPLICATION_JSON)
                    .headers(headers -> headers.setContentDispositionFormData("file", "patch.json"));

            String response = webClient.post()
                    .uri(uriBuilder.build().toUri())
                    .headers(headers -> {
                        headers.setContentType(MediaType.MULTIPART_FORM_DATA);
                        headers.set(HttpHeaders.AUTHORIZATION, String.format("OAuth %s", accessToken));
                    })
                    .body(BodyInserters.fromMultipartData(multipart.build()))
                    .retrieve()
                    .bodyToMono(String.class)
                    .onErrorMap(e -> {
                        LOGGER.error("Could not upload translations: {}", e.getMessage());
                        return new TankerClientException(e);
                    })
                    .block();

            LOGGER.debug("Content of Tanker response: {}", response);
        } catch (JsonProcessingException e) {
            LOGGER.error("Problem while encoding request payload: {}", e.getMessage());
            throw new TankerClientException(e);
        }
    }

    protected String encodeDocument(TankerDetailedJsonDoc tankerDetailedJsonDoc) throws JsonProcessingException {
        ObjectMapper objectMapper = new ObjectMapper();
        return objectMapper.writeValueAsString(tankerDetailedJsonDoc);
    }
}
