package ru.yandex.partner.libs.i18n.tanker.sync;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.List;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Options;
import org.springframework.util.StringUtils;
import org.springframework.web.reactive.function.client.WebClient;

import ru.yandex.partner.libs.cli.CliApp;
import ru.yandex.partner.libs.cli.CliAppException;
import ru.yandex.partner.libs.i18n.escaper.IdentityStringEscaper;
import ru.yandex.partner.libs.i18n.escaper.StringEscaper;
import ru.yandex.partner.libs.i18n.resolver.bundle.DashResourceBundleNameResolver;
import ru.yandex.partner.libs.i18n.resolver.bundle.ResourceBundleNameResolver;
import ru.yandex.partner.libs.i18n.resolver.key.GettextMessageKeyResolver;
import ru.yandex.partner.libs.i18n.tanker.client.TankerClient;
import ru.yandex.partner.libs.i18n.tanker.client.TankerClientException;
import ru.yandex.partner.libs.i18n.tanker.client.TankerClientImpl;
import ru.yandex.partner.libs.i18n.tanker.escaper.TankerPOStringEscaper;
import ru.yandex.partner.libs.i18n.tanker.resolver.TankerGettextMessageKeyResolver;
import ru.yandex.partner.libs.i18n.testing.MessagesCollector;
import ru.yandex.partner.libs.i18n.testing.MsgNotEnumException;
import ru.yandex.partner.libs.i18n.translation.source.factory.ResourceFileTranslationSourceFactory;
import ru.yandex.partner.libs.i18n.translation.source.factory.TranslationSourceFactory;

/**
 * Usage & details: https://wiki.yandex-team.ru/partner/w/dev/2java/i18n/#tanker-sync
 */
public class TankerSyncApp extends CliApp {

    private TranslationDownloader translationDownloader;
    private TranslationPatchGenerator translationPatchGenerator;
    private TranslationPatchUploader translationPatchUploader;
    private boolean doUpload;
    private List<String> languages;
    private Path patchFilePath;
    private String patchFileLanguage;

    public static void main(String[] args) throws Exception {
        (new TankerSyncApp()).run(args);
    }

    @Override
    protected Options getOptions() {
        Options options = new Options();
        for (TankerSyncCliOption opt : TankerSyncCliOption.values()) {
            options.addOption(opt.getOption());
        }
        return options;
    }

    @Override
    protected void configure(CommandLine cmd) throws CliAppException {
        doUpload = cmd.hasOption(TankerSyncCliOption.DO_UPLOAD.getOption().getLongOpt());
        String authToken = TankerSyncCliOption.TANKER_AUTH_TOKEN.resolveValue(cmd);

        if (doUpload && StringUtils.isEmpty(authToken)) {
            throw new CliAppException(
                    String.format("For --%s mode please specify Tanker auth token via --%s or %s environment variable. "
                                    + "To obtain token please visit https://nda.ya.ru/3SjQY7.",
                            TankerSyncCliOption.DO_UPLOAD.getOption().getLongOpt(),
                            TankerSyncCliOption.TANKER_AUTH_TOKEN.getOption().getLongOpt(),
                            TankerSyncCliOption.Constants.TANKER_AUTH_TOKEN_ENV_VARIABLE)
            );
        }

        String basePackage = TankerSyncCliOption.BASE_PACKAGE.resolveValue(cmd);
        MessagesCollector messagesCollector = new MessagesCollector(basePackage);

        String tankerBaseUrl = TankerSyncCliOption.TANKER_URL.resolveValue(cmd);
        String projectId = TankerSyncCliOption.TANKER_PROJECT_ID.resolveValue(cmd);
        String branchId = TankerSyncCliOption.TANKER_BRANCH_ID.resolveValue(cmd);
        TankerClient tankerClient =
                new TankerClientImpl(WebClient.create(), tankerBaseUrl, projectId, branchId, authToken);

        String baseResourcePath = TankerSyncCliOption.BASE_RESOURCE_PACKAGE.resolveValue(cmd);
        ResourceBundleNameResolver resourceBundleNameResolver = new DashResourceBundleNameResolver(baseResourcePath);

        String resourcesDirectoryStr = TankerSyncCliOption.RESOURCES_DIRECTORY.resolveValue(cmd);
        resourcesDirectoryStr = resourcesDirectoryStr.replaceFirst("^~", System.getProperty("user.home"));
        File resourceDirectory = new File(resourcesDirectoryStr);

        translationDownloader = new TranslationDownloader(
                messagesCollector, tankerClient, resourceDirectory, resourceBundleNameResolver);

        GettextMessageKeyResolver gettextMessageKeyResolver = new TankerGettextMessageKeyResolver();

        boolean noEscapeQuotes = cmd.hasOption(TankerSyncCliOption.NO_ESCAPE_QUOTES.getOption().getLongOpt());
        StringEscaper stringEscaper = noEscapeQuotes ? new IdentityStringEscaper() : new TankerPOStringEscaper();

        TranslationSourceFactory translationSourceFactory = new ResourceFileTranslationSourceFactory(
                resourceDirectory, resourceBundleNameResolver, gettextMessageKeyResolver, stringEscaper);

        translationPatchGenerator = new TranslationPatchGenerator(
                messagesCollector, translationSourceFactory, gettextMessageKeyResolver
        );

        translationPatchUploader = new TranslationPatchUploader(
                messagesCollector, translationSourceFactory, gettextMessageKeyResolver, stringEscaper, tankerClient
        );

        languages = TankerSyncCliOption.LANGUAGES.resolveValues(cmd);

        String patchFile = TankerSyncCliOption.PATCH_FILE.resolveValue(cmd);
        patchFilePath = Path.of(patchFile);

        patchFileLanguage = TankerSyncCliOption.PATCH_LANGUAGE.resolveValue(cmd);
    }

    @Override
    protected void run() throws IOException, MsgNotEnumException, TankerClientException {
        if (doUpload) {
            translationPatchUploader.uploadPatch(patchFileLanguage, patchFilePath);
        }
        translationDownloader.downloadTranlsations(languages);
        translationPatchGenerator.generatePatch(patchFileLanguage, patchFilePath);
    }


}
