package ru.yandex.partner.libs.i18n.tanker.sync;

import java.util.List;
import java.util.Optional;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;

public enum TankerSyncCliOption {
    DO_UPLOAD(new Option("u", "upload", false, "Use this option to upload patch to Tanker")),
    BASE_PACKAGE(new Option(null, "base-package", true, "Base package to scan for messages"), "ru.yandex.partner"),
    TANKER_URL(new Option(null, "tanker-url", true, "Base URL of tanker API"), "https://tanker-api.yandex-team.ru"),
    TANKER_PROJECT_ID(new Option("p", "project-id", true, "Tanker project ID"), "pi2"),
    TANKER_BRANCH_ID(new Option("b", "branch-id", true, "Tanker branch ID"), "master"),
    RESOURCES_DIRECTORY(new Option("d", "resources-dir", true, "Directory that store resources"), "src/main/resources"),
    NO_ESCAPE_QUOTES(new Option("q", "no-escape-quotes", false,
            "Indicates that quotes are not escaped with backslash in Tanker (escaped by default)")),
    BASE_RESOURCE_PACKAGE(new Option("r", "resource-package", true, "Base name for resource packages"), "translations"),
    LANGUAGES(Option.builder("l").longOpt("languages").desc("Languages to download"), List.of("en", "ru", "tr")),
    PATCH_FILE(new Option("f", "patch-file", true, "Translation patch file location"), "translation_patch.json"),
    PATCH_LANGUAGE(new Option("n", "patch-language", true, "Translation patch language"), "ru"),
    TANKER_AUTH_TOKEN(new Option("t", "auth-token", true, "Tanker authentication token"),
            System.getenv(Constants.TANKER_AUTH_TOKEN_ENV_VARIABLE));

    private final Option option;
    private final String defaultValue;
    private final List<String> defaultValues;

    TankerSyncCliOption(Option option, String defaultValue) {
        this.option = option;
        this.defaultValue = defaultValue;
        this.defaultValues = null;
    }

    TankerSyncCliOption(Option.Builder builder, List<String> defaultValues) {
        this.option = builder.hasArgs().valueSeparator(',').build();
        this.defaultValue = null;
        this.defaultValues = defaultValues;
    }

    TankerSyncCliOption(Option option) {
        this.option = option;
        this.defaultValue = null;
        this.defaultValues = null;
    }

    public Option getOption() {
        return option;
    }

    public String getDefaultValue() {
        return defaultValue;
    }

    public String resolveValue(CommandLine cmd) {
        return Optional.ofNullable(cmd.getOptionValue(getOption().getLongOpt())).orElse(getDefaultValue());
    }

    public List<String> getDefaultValues() {
        return defaultValues;
    }

    public List<String> resolveValues(CommandLine cmd) {
        return Optional.ofNullable(cmd.getOptionValues(getOption().getLongOpt()))
                .map(List::of).orElseGet(this::getDefaultValues);
    }

    public static class Constants {
        public static final String TANKER_AUTH_TOKEN_ENV_VARIABLE = "TANKER_AUTH_TOKEN";
    }
}
