package ru.yandex.partner.libs.i18n.tanker.sync;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.partner.libs.i18n.GettextMsg;
import ru.yandex.partner.libs.i18n.tanker.client.TankerClient;
import ru.yandex.partner.libs.i18n.tanker.client.TankerClientException;
import ru.yandex.partner.libs.i18n.tanker.dto.json.TankerJsonKeyset;
import ru.yandex.partner.libs.i18n.tanker.dto.json.TankerJsonKeysetDoc;
import ru.yandex.partner.libs.i18n.testing.MessagesCollector;
import ru.yandex.partner.libs.i18n.testing.MsgNotEnumException;
import ru.yandex.partner.libs.i18n.resolver.bundle.ResourceBundleNameResolver;

public class TranslationDownloader {

    private static final Logger LOGGER = LoggerFactory.getLogger(TranslationDownloader.class);

    private final MessagesCollector messagesCollector;
    private final TankerClient tankerClient;
    private final File resourceDirectory;
    private final ResourceBundleNameResolver resourceBundleNameResolver;

    public TranslationDownloader(MessagesCollector messagesCollector, TankerClient tankerClient,
                                 File resourceDirectory, ResourceBundleNameResolver resourceBundleNameResolver) {
        this.messagesCollector = messagesCollector;
        this.tankerClient = tankerClient;
        this.resourceDirectory = resourceDirectory;
        this.resourceBundleNameResolver = resourceBundleNameResolver;
    }

    public void downloadTranlsations(List<String> languages)
            throws MsgNotEnumException {
        Set<String> keysets = messagesCollector.collectMessages().stream()
                .map(GettextMsg::getKeysetName).collect(Collectors.toSet());

        record P(String lang, String keyset) { }

        languages.stream()
                .flatMap(l -> keysets.stream().map(k -> new P(l, k)))
                .parallel()
                .forEach(p -> {
                    try {
                        TankerJsonKeysetDoc tankerJsonKeysetDoc = downloadOrInitKeyset(p.keyset(), p.lang());
                        Path filePath = resourceBundleNameResolver
                                .resolveResourceFilePath(resourceDirectory, p.keyset(), p.lang());
                        saveTranslation(tankerJsonKeysetDoc, filePath);
                    } catch (TankerClientException | IOException e) {
                        throw new RuntimeException(e);
                    }
                });
    }

    private TankerJsonKeysetDoc downloadOrInitKeyset(String keyset, String language) throws TankerClientException {
        LOGGER.info("Downloading keyset '{}' for language '{}'", keyset, language);
        TankerJsonKeysetDoc tankerJsonKeysetDoc;
        tankerJsonKeysetDoc = tankerClient.downloadKeyset(keyset, language);
        if (tankerJsonKeysetDoc == null) {
            LOGGER.info("Could not download keyset '{}' for language '{}', creating empty resource", keyset, language);
            tankerJsonKeysetDoc = new TankerJsonKeysetDoc();
            tankerJsonKeysetDoc.addKeyset(language, new TankerJsonKeyset());
        }
        return tankerJsonKeysetDoc;
    }

    private void saveTranslation(TankerJsonKeysetDoc tankerJsonKeysetDoc, Path filePath) throws IOException {
        ObjectMapper objectMapper = new ObjectMapper();
        LOGGER.debug("Saving file '{}'", filePath);
        filePath.toFile().getParentFile().mkdirs();
        Files.write(filePath, objectMapper.writerWithDefaultPrettyPrinter()
                .writeValueAsBytes(tankerJsonKeysetDoc));
    }
}
