package ru.yandex.partner.libs.i18n.tanker.sync;

import java.io.BufferedReader;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.partner.libs.i18n.resolver.key.GettextMessageKeyResolver;
import ru.yandex.partner.libs.i18n.GettextMsg;
import ru.yandex.partner.libs.i18n.tanker.sync.dto.TranslationPatch;
import ru.yandex.partner.libs.i18n.tanker.sync.dto.TranslationPatchGroup;
import ru.yandex.partner.libs.i18n.resolver.plural.PluralVariantResolverProvider;
import ru.yandex.partner.libs.i18n.translation.source.MergingTranslationSource;
import ru.yandex.partner.libs.i18n.translation.source.PatchTranslationSource;
import ru.yandex.partner.libs.i18n.translation.source.TranslationSource;
import ru.yandex.partner.libs.i18n.translation.source.factory.TranslationSourceFactory;
import ru.yandex.partner.libs.i18n.testing.MessageChecker;
import ru.yandex.partner.libs.i18n.testing.MessagesCollector;
import ru.yandex.partner.libs.i18n.testing.MsgNotEnumException;
import ru.yandex.partner.libs.i18n.translation.Translation;

public class TranslationPatchGenerator {

    private static final Logger LOGGER = LoggerFactory.getLogger(TranslationPatchGenerator.class);

    private final MessagesCollector messagesCollector;
    private final TranslationSourceFactory translationSourceFactory;
    private final GettextMessageKeyResolver gettextMessageKeyResolver;

    public TranslationPatchGenerator(MessagesCollector messagesCollector,
                                     TranslationSourceFactory translationSourceFactory,
                                     GettextMessageKeyResolver gettextMessageKeyResolver) {
        this.messagesCollector = messagesCollector;
        this.translationSourceFactory = translationSourceFactory;
        this.gettextMessageKeyResolver = gettextMessageKeyResolver;
    }

    public void generatePatch(String patchLanguage, Path patchFilePath) throws MsgNotEnumException, IOException {

        MessageChecker messageChecker = new MessageChecker(messagesCollector, translationSourceFactory);

        LOGGER.info("Searching for missing or malformed translations for language {}", patchLanguage);

        Locale locale = new Locale(patchLanguage);
        Set<GettextMsg> missingTranslations = messageChecker.findProblematicTranslations(locale);

        if (missingTranslations.isEmpty()) {
            LOGGER.info("No missing translations found");
            if (Files.exists(patchFilePath)) {
                Files.delete(patchFilePath);
            }
            return;
        } else {
            LOGGER.warn(
                    "Found {} missing or malformed translations : {}",
                    missingTranslations.size(),
                    missingTranslations
            );
        }

        TranslationSource translationSource = getTranslationSource(patchFilePath, translationSourceFactory, locale);
        TranslationPatch patch = generatePatchObject(missingTranslations, translationSource);

        savePatch(patchFilePath, patch);
        LOGGER.info("Please fill {} with '{}' translations of messages and then run './tanker_sync.sh --upload'",
                patchFilePath.toString(), patchLanguage);

    }

    protected TranslationSource getTranslationSource(Path patchFilePath,
                                                     TranslationSourceFactory translationSourceFactory,
                                                     Locale locale) throws IOException {
        TranslationSource translationSource = translationSourceFactory.getTranslationSource(locale);

        if (Files.exists(patchFilePath)) {
            try (BufferedReader reader = Files.newBufferedReader(patchFilePath)) {
                ObjectMapper objectMapper = new ObjectMapper();
                TranslationPatch oldPatch = objectMapper.readValue(reader, TranslationPatch.class);
                TranslationSource patchTranslationSource = new PatchTranslationSource(oldPatch,
                        PluralVariantResolverProvider.getPluralVariantResolver(locale.getLanguage()),
                        gettextMessageKeyResolver);
                translationSource = new MergingTranslationSource(translationSource, patchTranslationSource);
            }
        }

        return translationSource;
    }

    protected TranslationPatch generatePatchObject(Set<GettextMsg> missingTranslations,
                                                   TranslationSource translationSource) {
        TranslationPatch patch = new TranslationPatch();

        Map<Class<? extends GettextMsg>, Set<GettextMsg>> missingTranslationsByClass = new HashMap<>();
        for (GettextMsg msg : missingTranslations) {
            missingTranslationsByClass.computeIfAbsent(msg.getClass(), k -> new HashSet<>()).add(msg);
        }

        for (Map.Entry<Class<? extends GettextMsg>, Set<GettextMsg>> classEntry :
                missingTranslationsByClass.entrySet()) {
            TranslationPatchGroup patchGroup = generateTranslationPatchGroup(translationSource, classEntry.getValue());
            patch.addGroup(classEntry.getKey().getCanonicalName(), patchGroup);
        }
        return patch;
    }

    private TranslationPatchGroup generateTranslationPatchGroup(TranslationSource translationSource,
                                                                Set<GettextMsg> messages) {
        TranslationPatchGroup patchGroup = new TranslationPatchGroup();
        for (GettextMsg gettextMsg : messages) {
            Translation translation = translationSource.getTranslation(gettextMsg);

            Object patchTranslation;
            if (!translation.isPlural()) {
                patchTranslation = translation.getTranslation();
            } else {
                patchTranslation = translation.getTranslationsAsMap();
            }
            String fullKey = gettextMessageKeyResolver.getMessageKey(gettextMsg.getPayload());

            patchGroup.addTranslation(fullKey, patchTranslation);
        }
        return patchGroup;
    }

    protected void savePatch(Path patchFilePath, TranslationPatch patch) throws IOException {
        ObjectMapper objectMapper = new ObjectMapper();
        Files.write(patchFilePath, objectMapper.writerWithDefaultPrettyPrinter()
                .writeValueAsBytes(patch));
    }

}
