package ru.yandex.partner.libs.i18n.tanker.sync;

import java.io.BufferedReader;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Locale;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.partner.libs.i18n.GettextMsg;
import ru.yandex.partner.libs.i18n.GettextMsgPayload;
import ru.yandex.partner.libs.i18n.escaper.StringEscaper;
import ru.yandex.partner.libs.i18n.resolver.key.GettextMessageKeyResolver;
import ru.yandex.partner.libs.i18n.resolver.plural.PluralVariantResolver;
import ru.yandex.partner.libs.i18n.resolver.plural.PluralVariantResolverProvider;
import ru.yandex.partner.libs.i18n.tanker.client.TankerClient;
import ru.yandex.partner.libs.i18n.tanker.client.TankerClientException;
import ru.yandex.partner.libs.i18n.tanker.dto.tjson.TankerDetailedJsonDoc;
import ru.yandex.partner.libs.i18n.tanker.dto.tjson.TankerDetailedJsonKey;
import ru.yandex.partner.libs.i18n.tanker.dto.tjson.TankerDetailedJsonKeyInfo;
import ru.yandex.partner.libs.i18n.tanker.dto.tjson.TankerDetailedJsonKeys;
import ru.yandex.partner.libs.i18n.tanker.dto.tjson.TankerDetailedJsonKeyset;
import ru.yandex.partner.libs.i18n.tanker.dto.tjson.TankerDetailedJsonKeysetMeta;
import ru.yandex.partner.libs.i18n.tanker.dto.tjson.TankerDetailedJsonKeysets;
import ru.yandex.partner.libs.i18n.tanker.dto.tjson.TankerDetailedJsonPluralTranslation;
import ru.yandex.partner.libs.i18n.tanker.dto.tjson.TankerDetailedJsonSingleTranslation;
import ru.yandex.partner.libs.i18n.tanker.dto.tjson.TankerDetailedJsonTranslation;
import ru.yandex.partner.libs.i18n.tanker.dto.tjson.TankerDetailedJsonTranslations;
import ru.yandex.partner.libs.i18n.tanker.resolver.TankerGettextMessageKeyResolver;
import ru.yandex.partner.libs.i18n.tanker.sync.dto.TranslationPatch;
import ru.yandex.partner.libs.i18n.testing.MessageChecker;
import ru.yandex.partner.libs.i18n.testing.MessagesCollector;
import ru.yandex.partner.libs.i18n.testing.MsgNotEnumException;
import ru.yandex.partner.libs.i18n.translation.Translation;
import ru.yandex.partner.libs.i18n.translation.source.PatchTranslationSource;
import ru.yandex.partner.libs.i18n.translation.source.TranslationSource;
import ru.yandex.partner.libs.i18n.translation.source.factory.TranslationSourceFactory;

public class TranslationPatchUploader {

    private static final Logger LOGGER = LoggerFactory.getLogger(TranslationPatchUploader.class);

    private final MessagesCollector messagesCollector;
    private final TranslationSourceFactory translationSourceFactory;
    private final GettextMessageKeyResolver gettextMessageKeyResolver;
    private final StringEscaper stringEscaper;
    private final TankerClient tankerClient;

    public TranslationPatchUploader(MessagesCollector messagesCollector,
                                    TranslationSourceFactory translationSourceFactory,
                                    GettextMessageKeyResolver gettextMessageKeyResolver,
                                    StringEscaper stringEscaper,
                                    TankerClient tankerClient) {
        this.messagesCollector = messagesCollector;
        this.translationSourceFactory = translationSourceFactory;
        this.gettextMessageKeyResolver = gettextMessageKeyResolver;
        this.stringEscaper = stringEscaper;
        this.tankerClient = tankerClient;
    }

    public void uploadPatch(String patchLanguage, Path patchFilePath) throws MsgNotEnumException, IOException,
            TankerClientException {

        if (!Files.exists(patchFilePath)) {
            LOGGER.info("File {} does not exist, skipping uploading translations", patchFilePath);
            return;
        }

        MessageChecker messageChecker = new MessageChecker(messagesCollector, translationSourceFactory);
        Set<GettextMsg> problematicTranslations =
                messageChecker.findProblematicTranslations(new Locale(patchLanguage));

        if (problematicTranslations.isEmpty()) {
            LOGGER.info("All translations exist in resources, skipping uploading translations");
            return;
        }

        try (BufferedReader reader = Files.newBufferedReader(patchFilePath)) {
            LOGGER.info("Reading {}", patchFilePath);

            ObjectMapper objectMapper = new ObjectMapper();
            TranslationPatch patch = objectMapper.readValue(reader, TranslationPatch.class);

            LOGGER.info("Uploading contents of {} to Tanker", patchFilePath);

            TankerDetailedJsonDoc tankerDetailedJsonDoc =
                    generateTJsonDocument(patch, patchLanguage, problematicTranslations);

            tankerClient.uploadTranslations(tankerDetailedJsonDoc);
        }

        LOGGER.info("Patch uploaded, deleting {}", patchFilePath);
        Files.delete(patchFilePath);

    }

    protected TankerDetailedJsonDoc generateTJsonDocument(TranslationPatch patch, String patchLanguage,
                                                          Set<GettextMsg> problematicTranslations) {
        TankerDetailedJsonDoc tankerDetailedJsonDoc = new TankerDetailedJsonDoc();

        PluralVariantResolver pluralVariantResolver =
                PluralVariantResolverProvider.getPluralVariantResolver(patchLanguage);


        addAllTranslationToTJsonDoc(tankerDetailedJsonDoc, patchLanguage,
                problematicTranslations,
                new PatchTranslationSource(patch, pluralVariantResolver, new TankerGettextMessageKeyResolver()));
        return tankerDetailedJsonDoc;
    }

    private void addAllTranslationToTJsonDoc(TankerDetailedJsonDoc tankerDetailedJsonDoc, String language,
                                             Set<GettextMsg> messages,
                                             TranslationSource source) {
        for (GettextMsg msg : messages) {
            addTranslationToTJsonDoc(tankerDetailedJsonDoc, language, msg, source);
        }
    }

    private void addTranslationToTJsonDoc(TankerDetailedJsonDoc tankerDetailedJsonDoc, String language,
                                          GettextMsg gettextMsg,
                                          TranslationSource source) {
        if (tankerDetailedJsonDoc.getKeysets() == null) {
            tankerDetailedJsonDoc.setKeysets(new TankerDetailedJsonKeysets());
        }
        TankerDetailedJsonKeyset tankerDetailedJsonKeyset =
                tankerDetailedJsonDoc.getKeysets().getKeysets().computeIfAbsent(gettextMsg.getKeysetName(),
                        s -> new TankerDetailedJsonKeyset());
        if (tankerDetailedJsonKeyset.getMeta() == null) {
            tankerDetailedJsonKeyset.setMeta(new TankerDetailedJsonKeysetMeta(new ArrayList<>()));
        }
        if (!tankerDetailedJsonKeyset.getMeta().getLanguages().contains(language)) {
            tankerDetailedJsonKeyset.getMeta().getLanguages().add(language);
        }
        GettextMsgPayload payload = gettextMsg.getPayload();
        String fullKey = gettextMessageKeyResolver.getMessageKey(payload);
        String escapedKey = stringEscaper.escapeString(fullKey);
        if (tankerDetailedJsonKeyset.getKeys() == null) {
            tankerDetailedJsonKeyset.setKeys(new TankerDetailedJsonKeys());
        }
        TankerDetailedJsonKey tankerDetailedJsonKey = tankerDetailedJsonKeyset.getKeys().getKeys()
                .computeIfAbsent(escapedKey, s -> new TankerDetailedJsonKey());
        if (tankerDetailedJsonKey.getInfo() == null) {
            tankerDetailedJsonKey.setInfo(new TankerDetailedJsonKeyInfo(payload.isPlural()));
        }
        if (tankerDetailedJsonKey.getTranslations() == null) {
            tankerDetailedJsonKey.setTranslations(new TankerDetailedJsonTranslations());
        }
        Translation translation = source.getTranslation(gettextMsg);
        TankerDetailedJsonTranslation tankerDetailedJsonTranslation = !translation.isPlural()
                ? generateSingleTranslation(translation)
                : generatePluralTranslation(translation);
        tankerDetailedJsonKey.getTranslations().addTranslation(language, tankerDetailedJsonTranslation);
    }

    private TankerDetailedJsonSingleTranslation generateSingleTranslation(Translation translation) {
        return new TankerDetailedJsonSingleTranslation(stringEscaper.escapeString(translation.getTranslation()));
    }

    private TankerDetailedJsonPluralTranslation generatePluralTranslation(Translation translation) {
        return new TankerDetailedJsonPluralTranslation(translation.getTranslations().stream()
                .map(stringEscaper::escapeString).collect(Collectors.toList()));
    }

}
