package ru.yandex.partner.libs.i18n.translation;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import ru.yandex.partner.libs.i18n.resolver.plural.PluralVariantResolver;
import ru.yandex.partner.libs.i18n.resolver.plural.PluralVariantResolverProvider;

public class Translation {

    private static final String EMPTY_STRING = "";

    private static final Translation EMPTY_SINGLE_TRANSLATION = new Translation(EMPTY_STRING);
    private static final Translation EMPTY_SINGLE_MALFORMED_TRANSLATION = new Translation(EMPTY_STRING, true);
    private static final Map<Integer, List<String>> EMPTY_TRANSLATION_LISTS = new HashMap<>();

    private final List<String> translations;
    private final PluralVariantResolver pluralVariantResolver;
    private final boolean resourceMalformed;

    public Translation(String translationString, boolean resourceMalformed) {
        translations = List.of(translationString != null ? translationString : EMPTY_STRING);
        this.pluralVariantResolver = null;
        this.resourceMalformed = resourceMalformed;
    }

    public Translation(String translationString) {
        this(translationString, false);
    }

    public Translation(List<?> translations, PluralVariantResolver pluralVariantResolver, boolean resourceMalformed) {
        this.pluralVariantResolver = pluralVariantResolver;
        this.translations = new ArrayList<>(pluralVariantResolver.variantsCount());
        for (int i = 0; i < pluralVariantResolver.variantsCount(); i++) {
            this.translations.add(translations.size() > i && translations.get(i) instanceof String
                    ? (String) translations.get(i) : EMPTY_STRING);
        }
        this.resourceMalformed = resourceMalformed;
    }

    public Translation(List<?> translations, PluralVariantResolver pluralVariantResolver) {
        this(translations, pluralVariantResolver, false);
    }

    private Translation(PluralVariantResolver pluralVariantResolver, boolean resourceMalformed) {
        int translationsCount = pluralVariantResolver.variantsCount();
        this.translations = EMPTY_TRANSLATION_LISTS.computeIfAbsent(translationsCount,
                count -> Collections.nCopies(count, EMPTY_STRING));
        this.pluralVariantResolver = pluralVariantResolver;
        this.resourceMalformed = resourceMalformed;
    }

    public Translation(List<Object> translations, String language) {
        this(translations, PluralVariantResolverProvider.getPluralVariantResolver(language));
    }

    public String getTranslation() {
        return translations.get(0);
    }

    public List<String> getTranslations() {
        return new ArrayList<>(translations);
    }

    public LinkedHashMap<String, String> getTranslationsAsMap() {
        if (!isPlural()) {
            return new LinkedHashMap<>(Map.of(EMPTY_STRING, translations.get(0)));
        }
        LinkedHashMap<String, String> m = new LinkedHashMap<>();
        for (int i = 0; i < translations.size(); i++) {
            List<String> variantDescriptions = pluralVariantResolver.getVariantDescriptions();
            m.put(variantDescriptions.get(i), translations.get(i));
        }
        return m;
    }

    public String getTranslationForNumber(int n) {
        if (!isPlural()) {
            return getTranslation();
        }
        return translations.get(pluralVariantResolver.resolvePluralVariant(n));
    }

    public PluralVariantResolver getPluralVariantResolver() {
        return pluralVariantResolver;
    }

    public boolean isPlural() {
        return pluralVariantResolver != null;
    }

    public boolean hasMissingTranslations() {
        return translations.stream().anyMatch(EMPTY_STRING::equals);
    }

    public boolean isResourceMalformed() {
        return resourceMalformed;
    }

    public static Translation emptySingleTranslation() {
        return EMPTY_SINGLE_TRANSLATION;
    }

    public static Translation getEmptySingleMalformedTranslation() {
        return EMPTY_SINGLE_MALFORMED_TRANSLATION;
    }

    public static Translation emptyPluralTranslation(PluralVariantResolver pluralVariantResolver) {
        return new Translation(pluralVariantResolver, false);
    }

    public static Translation emptyPluralMalformedTranslation(PluralVariantResolver pluralVariantResolver) {
        return new Translation(pluralVariantResolver, true);
    }

}
