package ru.yandex.partner.libs.i18n.translation.source;

import java.util.Collections;
import java.util.HashSet;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.partner.libs.i18n.GettextMsg;
import ru.yandex.partner.libs.i18n.GettextMsgPayload;
import ru.yandex.partner.libs.i18n.resolver.bundle.ResourceBundleNameResolver;
import ru.yandex.partner.libs.i18n.resolver.key.GettextMessageKeyResolver;

public class ResourceBundleTranslationSource extends BaseResourceTranslationSource {

    private final static Logger LOGGER = LoggerFactory.getLogger(ResourceBundleTranslationSource.class);

    private final Locale locale;
    private final GettextMessageKeyResolver gettextMessageKeyResolver;
    private final ResourceBundle.Control resourceBundleControl;
    private final ResourceBundleNameResolver resourceBundleNameResolver;

    public ResourceBundleTranslationSource(Locale locale,
                                           ResourceBundle.Control resourceBundleControl,
                                           GettextMessageKeyResolver gettextMessageKeyResolver,
                                           ResourceBundleNameResolver resourceBundleNameResolver) {
        this.resourceBundleNameResolver = resourceBundleNameResolver;
        this.locale = locale;
        this.gettextMessageKeyResolver = gettextMessageKeyResolver;
        this.resourceBundleControl = resourceBundleControl;
    }

    @Override
    protected ResourceRetrievalResult retrieveResource(GettextMsg gettextMsg) {
        GettextMsgPayload payload = gettextMsg.getPayload();
        String fullBundleName = resourceBundleNameResolver.resolveResourceBundleName(gettextMsg.getKeysetName());

        ResourceBundle resourceBundle = getResourceBundle(fullBundleName, locale);
        if (resourceBundle == null) {
            return new ResourceRetrievalResult(null, locale);
        }

        Set<String> existingKeys = new HashSet<>(Collections.list(resourceBundle.getKeys()));
        String key = gettextMessageKeyResolver.getMessageKey(payload);
        if (!existingKeys.contains(key)) {
            return new ResourceRetrievalResult(null, resourceBundle.getLocale());
        }

        return new ResourceRetrievalResult(resourceBundle.getObject(key), resourceBundle.getLocale());
    }

    private ResourceBundle getResourceBundle(String fullBundleName, Locale locale) {
        try {
            return ResourceBundle.getBundle(fullBundleName, locale, resourceBundleControl);
        } catch (MissingResourceException e) {
            LOGGER.error("Error loading resource bundle", e);
            return null;
        }
    }

}
