package ru.yandex.partner.libs.i18n.translation.source;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.partner.libs.i18n.GettextMsg;
import ru.yandex.partner.libs.i18n.GettextMsgPayload;
import ru.yandex.partner.libs.i18n.escaper.StringEscaper;
import ru.yandex.partner.libs.i18n.resolver.bundle.ResourceBundleNameResolver;
import ru.yandex.partner.libs.i18n.resolver.key.GettextMessageKeyResolver;
import ru.yandex.partner.libs.i18n.tanker.dto.json.TankerJsonKeysetDoc;

public class ResourceFileTranslationSource extends BaseResourceTranslationSource {

    private final static Logger LOGGER = LoggerFactory.getLogger(ResourceFileTranslationSource.class);

    private final File resourceDirectory;
    private final ResourceBundleNameResolver resourceBundleNameResolver;
    private final GettextMessageKeyResolver gettextMessageKeyResolver;
    private final StringEscaper stringEscaper;
    private final Locale locale;

    private final Map<String, TankerJsonKeysetDoc> parsedResources = new ConcurrentHashMap<>();

    public ResourceFileTranslationSource(File resourceDirectory,
                                         ResourceBundleNameResolver resourceBundleNameResolver,
                                         GettextMessageKeyResolver gettextMessageKeyResolver,
                                         StringEscaper stringEscaper, Locale locale) {
        this.resourceDirectory = resourceDirectory;
        this.resourceBundleNameResolver = resourceBundleNameResolver;
        this.gettextMessageKeyResolver = gettextMessageKeyResolver;
        this.stringEscaper = stringEscaper;
        this.locale = locale;
    }

    @Override
    protected ResourceRetrievalResult retrieveResource(GettextMsg gettextMsg) {
        String keysetName = gettextMsg.getKeysetName();
        GettextMsgPayload payload = gettextMsg.getPayload();
        String language = locale.getLanguage();

        TankerJsonKeysetDoc tankerJsonKeysetDoc = getTranslationLanguage(keysetName);

        if (tankerJsonKeysetDoc == null || !tankerJsonKeysetDoc.getKeysetEntries().containsKey(language)) {
            return new ResourceRetrievalResult(null, locale);
        }

        String fullKey = gettextMessageKeyResolver.getMessageKey(payload);
        Object resource = getResource(tankerJsonKeysetDoc, language, fullKey);

        return new ResourceRetrievalResult(resource, locale);
    }

    private Object getResource(TankerJsonKeysetDoc tankerJsonKeysetDoc, String language, String fullKey) {
        String escapedKey = stringEscaper.escapeString(fullKey);
        return stringEscaper.unsecapeResource(
                tankerJsonKeysetDoc.getKeysetEntries().get(language).getTranslationEntries().get(escapedKey)
        );
    }

    private TankerJsonKeysetDoc getTranslationLanguage(String keyset) {
        return parsedResources.computeIfAbsent(keyset, this::loadTranslationLanguage);
    }

    public TankerJsonKeysetDoc loadTranslationLanguage(String keyset) {
        Path filePath = resourceBundleNameResolver.resolveResourceFilePath(resourceDirectory, keyset,
                locale.getLanguage());
        try (BufferedReader reader = Files.newBufferedReader(filePath)) {
            ObjectMapper objectMapper = new ObjectMapper();
            return objectMapper.readValue(reader, TankerJsonKeysetDoc.class);
        } catch (IOException e) {
            LOGGER.warn("Could not read or decode {}: {}", filePath, e.getMessage());
            return null;
        }
    }


}
